(function() {
    'use strict';
    console.log('visualizer.js v1.6.0 loaded');

    /* ============================================================
       SECTION 1: STATE
       ============================================================ */
    var state = {
        currentStep: 1,
        imageFile: null,
        imageOrientation: null,
        answers: { size: null, features: [], finish: null, style: null },
        generatedImage: null,
        originalPreview: null,
        priceRange: null,
        retryCount: 0,
        isPopupMode: false,
        hasUsedVisualizer: false
    };

    /* ============================================================
       SECTION 2: STEP NAVIGATION (GPU-animated)
       ============================================================ */
    function goToStep(stepNum) {
        var current = document.querySelector('.viz-step.active');
        var next = document.getElementById('vizStep' + stepNum);
        if (!next) return;

        if (current) {
            current.classList.add('exit-left');
            setTimeout(function() {
                current.classList.remove('active', 'exit-left');
            }, 400);
        }

        setTimeout(function() {
            next.classList.add('active');
            state.currentStep = stepNum;
            updateProgress(stepNum);

            if (state.isPopupMode) {
                // Scroll the flow popup inner container to top
                var inner = document.getElementById('vizFlowInner');
                if (inner) inner.scrollTop = 0;
            } else {
                var section = document.getElementById('calculator');
                if (section) section.scrollIntoView({ behavior: 'smooth', block: 'start' });
            }
        }, current ? 200 : 0);
    }

    function updateProgress(step) {
        var pct = Math.min(((step - 1) / 4) * 100, 100);
        var bar = document.getElementById('vizProgressBar');
        if (bar) bar.style.setProperty('--progress', pct + '%');

        var steps = document.querySelectorAll('.viz-progress-step');
        for (var i = 0; i < steps.length; i++) {
            var s = parseInt(steps[i].dataset.step);
            steps[i].classList.toggle('active', s === step);
            steps[i].classList.toggle('completed', s < step);
        }

        var prog = document.getElementById('vizProgress');
        if (prog) prog.style.display = step > 5 ? 'none' : '';
    }

    /* ============================================================
       SECTION 3: FILE UPLOAD
       ============================================================ */
    function initUpload() {
        var dropZone = document.getElementById('vizDropZone');
        var fileInput = document.getElementById('vizFileInput');
        var changeBtn = document.getElementById('vizChangePhoto');
        if (!dropZone || !fileInput) return;

        dropZone.addEventListener('click', function(e) {
            if (e.target.closest('.viz-change-photo') || e.target.closest('.viz-upload-preview') || e.target.closest('.viz-upload-btn')) return;
            fileInput.click();
        });

        fileInput.addEventListener('change', function(e) {
            if (e.target.files[0]) handleFile(e.target.files[0]);
        });

        ['dragenter', 'dragover'].forEach(function(evt) {
            dropZone.addEventListener(evt, function(e) {
                e.preventDefault();
                dropZone.classList.add('dragover');
            });
        });
        ['dragleave', 'drop'].forEach(function(evt) {
            dropZone.addEventListener(evt, function(e) {
                e.preventDefault();
                dropZone.classList.remove('dragover');
            });
        });
        dropZone.addEventListener('drop', function(e) {
            var file = e.dataTransfer.files[0];
            if (file) handleFile(file);
        });

        if (changeBtn) {
            changeBtn.addEventListener('click', function(e) {
                e.stopPropagation();
                fileInput.value = '';
                fileInput.click();
            });
        }
    }

    function handleFile(file) {
        hideError();
        validateImage(file).then(function(result) {
            if (!result.valid) {
                showError(result.error);
                return;
            }

            state.imageFile = file;
            state.imageOrientation = result.orientation;

            var reader = new FileReader();
            reader.onload = function(e) {
                state.originalPreview = e.target.result;
                document.getElementById('vizPreviewImg').src = e.target.result;
                document.getElementById('vizPreview').style.display = 'block';
                document.getElementById('vizUploadContent').style.display = 'none';
                document.getElementById('vizUploadSuccess').style.display = 'block';
            };
            reader.readAsDataURL(file);
        });
    }

    // Client-side validation (convenience - server validates again with 7 layers)
    function validateImage(file) {
        return new Promise(function(resolve) {
            if (file.size > 10 * 1024 * 1024) {
                return resolve({ valid: false, error: 'Image too large. Maximum size is 10MB.' });
            }
            if (['image/jpeg', 'image/png', 'image/webp'].indexOf(file.type) === -1) {
                return resolve({ valid: false, error: 'Please upload a JPG, PNG, or WebP image.' });
            }
            var img = new Image();
            var objUrl = URL.createObjectURL(file);
            img.onload = function() {
                URL.revokeObjectURL(objUrl);
                if (img.width < 400 || img.height < 300) {
                    return resolve({ valid: false, error: 'Image too small. Minimum 400x300 pixels.' });
                }
                resolve({
                    valid: true,
                    width: img.width,
                    height: img.height,
                    orientation: img.width >= img.height ? 'landscape' : 'portrait'
                });
            };
            img.onerror = function() {
                URL.revokeObjectURL(objUrl);
                resolve({ valid: false, error: 'Cannot read image. Please try a different file.' });
            };
            img.src = objUrl;
        });
    }

    function showError(msg) {
        var el = document.getElementById('vizUploadError');
        var text = document.getElementById('vizErrorText');
        if (el && text) {
            text.textContent = msg;
            el.style.display = 'block';
        }
    }

    function hideError() {
        var el = document.getElementById('vizUploadError');
        if (el) el.style.display = 'none';
    }

    /* ============================================================
       SECTION 4: CLIENT-SIDE IMAGE COMPRESSION (PERFORMANCE)
       Compresses to max 1600px wide, JPEG 0.85 before upload.
       Reduces upload payload from ~5MB to ~200-400KB.
       ============================================================ */
    function compressImage(file, maxWidth, quality) {
        maxWidth = maxWidth || 1600;
        quality = quality || 0.85;
        return new Promise(function(resolve) {
            var img = new Image();
            var objUrl = URL.createObjectURL(file);
            img.onload = function() {
                URL.revokeObjectURL(objUrl);
                var canvas = document.createElement('canvas');
                var w = img.width;
                var h = img.height;
                if (w > maxWidth) {
                    h = Math.round(h * (maxWidth / w));
                    w = maxWidth;
                }
                canvas.width = w;
                canvas.height = h;
                canvas.getContext('2d').drawImage(img, 0, 0, w, h);
                canvas.toBlob(resolve, 'image/jpeg', quality);
            };
            img.src = objUrl;
        });
    }

    /* ============================================================
       SECTION 5: QUIZ LOGIC (Steps 2-5)
       ============================================================ */
    function initQuiz() {
        // Size (Step 2) - auto-advance
        var sizeOpts = document.querySelectorAll('#vizStep2 .viz-option');
        for (var i = 0; i < sizeOpts.length; i++) {
            sizeOpts[i].addEventListener('click', function() {
                selectSingle(this, '#vizStep2');
                state.answers.size = this.dataset.value;
                setTimeout(function() { goToStep(3); }, 300);
            });
        }

        // Features (Step 3) - multi-select
        var featOpts = document.querySelectorAll('#vizStep3 .viz-option');
        for (var j = 0; j < featOpts.length; j++) {
            featOpts[j].addEventListener('click', function() {
                this.classList.toggle('selected');
                var selected = document.querySelectorAll('#vizStep3 .viz-option.selected');
                state.answers.features = [];
                for (var k = 0; k < selected.length; k++) {
                    state.answers.features.push(selected[k].dataset.value);
                }
                var cb = document.getElementById('vizFeatContinue');
                if (cb) cb.disabled = state.answers.features.length === 0;
            });
        }
        var featCont = document.getElementById('vizFeatContinue');
        if (featCont) {
            featCont.addEventListener('click', function() { goToStep(4); });
        }

        // Finish (Step 4) - auto-advance
        var finishOpts = document.querySelectorAll('#vizStep4 .viz-option');
        for (var m = 0; m < finishOpts.length; m++) {
            finishOpts[m].addEventListener('click', function() {
                selectSingle(this, '#vizStep4');
                state.answers.finish = this.dataset.value;
                setTimeout(function() { goToStep(5); }, 300);
            });
        }

        // Style (Step 5) - auto-advance + trigger generation
        var styleOpts = document.querySelectorAll('#vizStep5 .viz-option');
        for (var n = 0; n < styleOpts.length; n++) {
            styleOpts[n].addEventListener('click', function() {
                selectSingle(this, '#vizStep5');
                state.answers.style = this.dataset.value;
                setTimeout(function() {
                    goToStep(6);
                    startGeneration();
                }, 300);
            });
        }

        // Back buttons
        var backBtns = document.querySelectorAll('.viz-back-btn');
        for (var b = 0; b < backBtns.length; b++) {
            backBtns[b].addEventListener('click', function() {
                goToStep(parseInt(this.dataset.back));
            });
        }

        // Start Designing
        var startBtn = document.getElementById('vizStartDesign');
        if (startBtn) {
            startBtn.addEventListener('click', function() {
                state.hasUsedVisualizer = true;
                goToStep(2);
            });
        }
    }

    function selectSingle(clicked, containerSelector) {
        var opts = document.querySelectorAll(containerSelector + ' .viz-option');
        for (var i = 0; i < opts.length; i++) {
            opts[i].classList.remove('selected');
        }
        clicked.classList.add('selected');
    }

    /* ============================================================
       SECTION 6: API CALL + LOADING ANIMATION
       ============================================================ */
    var loadingMessages = (aceViz.text && aceViz.text.loadingMessages && aceViz.text.loadingMessages.length)
        ? aceViz.text.loadingMessages
        : [
            'Analyzing your backyard layout...',
            'Mapping property boundaries...',
            'Selecting your design elements...',
            'Applying your style preferences...',
            'Rendering photorealistic design...',
            'Adding landscape details...',
            'Fine-tuning lighting and shadows...',
            'Almost ready...'
        ];
    var loadingTips = (aceViz.text && aceViz.text.loadingTips && aceViz.text.loadingTips.length)
        ? aceViz.text.loadingTips
        : [
            'A swimming pool can add 5-8% to your home value',
            'Professional landscaping returns 100-200% ROI at resale',
            'Outdoor kitchens are the #1 requested feature in LA',
            'LED landscape lighting reduces energy costs by 75%',
            'ACE has completed over 500 backyard projects in Los Angeles',
            'The average ROI on a complete backyard remodel is 150%'
        ];

    var loadingInterval = null;
    var tipInterval = null;
    var progressInterval = null;
    var currentGenId = 0;

    function startLoading() {
        var msgIdx = 0;
        var tipIdx = 0;
        var progress = 0;
        var msgEl = document.getElementById('vizLoadMsg');
        var tipEl = document.getElementById('vizLoadTip');
        var barEl = document.getElementById('vizLoadBar');
        var pctEl = document.getElementById('vizLoadPercent');

        // Reset
        if (msgEl) { msgEl.textContent = loadingMessages[0]; msgEl.style.opacity = 1; }
        if (tipEl) { tipEl.textContent = loadingTips[0]; tipEl.style.opacity = 1; }
        if (barEl) barEl.style.width = '0%';
        if (pctEl) pctEl.textContent = '0%';

        // Messages every 3.5s
        loadingInterval = setInterval(function() {
            msgIdx = (msgIdx + 1) % loadingMessages.length;
            if (msgEl) {
                msgEl.style.opacity = 0;
                setTimeout(function() {
                    msgEl.textContent = loadingMessages[msgIdx];
                    msgEl.style.opacity = 1;
                }, 200);
            }
        }, 3500);

        // Tips every 6s
        tipInterval = setInterval(function() {
            tipIdx = (tipIdx + 1) % loadingTips.length;
            if (tipEl) {
                tipEl.style.opacity = 0;
                setTimeout(function() {
                    tipEl.textContent = loadingTips[tipIdx];
                    tipEl.style.opacity = 1;
                }, 200);
            }
        }, 6000);

        // Progress: 0-90% in ~15s (CSS transition handles smoothness)
        progressInterval = setInterval(function() {
            if (progress < 90) {
                progress += (90 - progress) * 0.05;
                if (barEl) barEl.style.width = progress + '%';
                if (pctEl) pctEl.textContent = Math.round(progress) + '%';
            }
        }, 300);
    }

    function stopLoading() {
        clearInterval(loadingInterval);
        clearInterval(tipInterval);
        clearInterval(progressInterval);
        loadingInterval = null;
        tipInterval = null;
        progressInterval = null;
        var barEl = document.getElementById('vizLoadBar');
        var pctEl = document.getElementById('vizLoadPercent');
        if (barEl) barEl.style.width = '100%';
        if (pctEl) pctEl.textContent = '100%';
    }

    function startGeneration() {
        var genId = ++currentGenId;
        startLoading();

        // PERF: compress image client-side before upload (1600px max, quality 0.85)
        compressImage(state.imageFile, 1600, 0.85).then(function(compressedBlob) {
            var compressedFile = new File([compressedBlob], 'backyard.jpg', { type: 'image/jpeg' });

            // Send compressed file via FormData (NOT base64)
            var formData = new FormData();
            formData.append('action', 'ace_generate_design');
            formData.append('nonce', aceViz.nonce);
            formData.append('image', compressedFile);
            formData.append('size', state.answers.size);
            formData.append('features', JSON.stringify(state.answers.features));
            formData.append('finish', state.answers.finish);
            formData.append('style', state.answers.style);
            // honeypot field left empty intentionally - bots will fill it

            return fetch(aceViz.ajax_url, { method: 'POST', body: formData });
        }).then(function(response) {
            return response.json();
        }).then(function(result) {
            if (genId !== currentGenId) return; // stale generation (popup closed mid-flight)
            stopLoading();

            if (result.success) {
                state.generatedImage = result.data.image_url;
                state.priceRange = { low: result.data.price_low, high: result.data.price_high };

                // Silently return to Step 5 under the popup
                var s6 = document.getElementById('vizStep6');
                var s5 = document.getElementById('vizStep5');
                if (s6) s6.classList.remove('active', 'exit-left');
                if (s5) s5.classList.add('active');
                state.currentStep = 5;
                updateProgress(5);

                // Exit popup mode to show result on main page
                if (state.isPopupMode) {
                    exitPopupToInline();
                }
                if (aceViz.settings.leadGate === 'none') {
                    showResult();
                } else {
                    showLeadGate(result.data.image_url);
                }
            } else {
                var msg = result.data && result.data.message ? result.data.message : 'Something went wrong.';
                showGenerationError(msg);
            }
        }).catch(function(error) {
            if (genId !== currentGenId) return;
            stopLoading();
            showGenerationError('Something went wrong. Please try again.');
        });
    }

    function showGenerationError(message) {
        goToStep(5);

        // Remove any existing error
        var existing = document.querySelector('#vizStep5 .viz-upload-error');
        if (existing) existing.remove();

        var errDiv = document.createElement('div');
        errDiv.className = 'viz-upload-error';
        errDiv.style.display = 'block';

        var p = document.createElement('p');
        p.textContent = message + ' ';

        var retryBtn = document.createElement('button');
        retryBtn.type = 'button';
        retryBtn.className = 'viz-retry-btn';
        retryBtn.textContent = 'Try Again';
        retryBtn.addEventListener('click', function() { errDiv.remove(); });

        p.appendChild(retryBtn);
        errDiv.appendChild(p);
        document.getElementById('vizStep5').appendChild(errDiv);
    }

    /* ============================================================
       SECTION 7: RESULT POPUP MODAL (dual-mode: overlay or inline)
       ============================================================ */
    var popupOpen = false;
    var gateVisible = false;

    /** Reset gate/result views to initial state */
    function resetPopupViews() {
        var gate = document.getElementById('vizPopupGate');
        var result = document.getElementById('vizPopupResult');
        if (gate) { gate.style.display = ''; gate.style.opacity = ''; gate.style.transition = ''; }
        if (result) { result.style.display = 'none'; result.style.opacity = ''; result.style.transition = ''; }
        var form = document.getElementById('vizGateForm');
        if (form) form.reset();
        var btn = form ? form.querySelector('.viz-btn-unlock') : null;
        if (btn) {
            btn.disabled = false;
            btn.innerHTML = '<i class="fas fa-lock-open"></i> Unlock My Design';
        }
        var emailAlt = document.getElementById('vizEmailAlt');
        if (emailAlt) emailAlt.style.display = 'none';
    }

    function openPopup() {
        if (state.isPopupMode) {
            // Popup mode: show result content inline within flow popup (no nested overlay)
            var container = document.querySelector('.viz-container');
            var vizPopup = document.getElementById('vizPopup');
            if (container) container.style.display = 'none';
            if (vizPopup) {
                vizPopup.style.display = 'block';
                vizPopup.classList.add('visible');
            }
            popupOpen = true;
            var inner = document.getElementById('vizFlowInner');
            if (inner) inner.scrollTop = 0;
            return;
        }

        // Inline mode: show result embedded in page section
        var popup = document.getElementById('vizPopup');
        if (!popup) return;
        var container = document.querySelector('.viz-container');
        if (container) {
            container.classList.add('viz-showing-result');
            container.appendChild(popup);
        }
        popup.style.display = 'block';
        popup.classList.add('flow-inline', 'visible');
        popupOpen = true;
        document.body.style.overflow = '';
        var sec = document.getElementById('calculator');
        if (sec) setTimeout(function(){ sec.scrollIntoView({behavior:'smooth'}); }, 100);
    }

    function closePopup(targetStep) {
        popupOpen = false;
        gateVisible = false;
        clearTimeout(gateTimer20);
        clearTimeout(gateTimer40);

        if (state.isPopupMode) {
            // Popup mode: hide result, show steps again
            var vizPopup = document.getElementById('vizPopup');
            var container = document.querySelector('.viz-container');

            if (vizPopup) {
                vizPopup.classList.remove('flow-inline', 'visible');
                vizPopup.style.display = 'none';
            }
            resetPopupViews();
            if (container) container.style.display = '';
            goToStep(targetStep || 5);
            return;
        }

        // Inline mode: hide result, restore quiz
        var popup = document.getElementById('vizPopup');
        if (!popup) return;
        popup.classList.remove('flow-inline', 'visible');
        popup.style.display = 'none';
        resetPopupViews();
        var container = document.querySelector('.viz-container');
        if (container) {
            container.style.display = '';
            container.classList.remove('viz-showing-result');
            container.parentNode.insertBefore(popup, container.nextSibling);
        }
        document.body.style.overflow = '';
        goToStep(targetStep || 5);
    }

    function initPopup() {
        // Close button (result popup)
        var closeBtn = document.getElementById('vizPopupClose');
        if (closeBtn) {
            closeBtn.addEventListener('click', function() { closePopup(5); });
        }

        // ESC key — handles result popup, flow popup, and promo popup
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') {
                if (popupOpen) {
                    // Result popup is showing (inline or popup mode)
                    closePopup(state.isPopupMode ? 2 : 5);
                } else if (state.isPopupMode) {
                    // Flow popup is open, no result showing — close entire flow
                    closeVisualizerPopup();
                } else {
                    // Maybe promo popup
                    var promo = document.getElementById('vizPromoPopup');
                    if (promo && promo.classList.contains('visible')) closePromoPopup();
                }
            }
        });

        // Backdrop click on result popup scroll wrapper (only in inline mode)
        var scroll = document.querySelector('.viz-popup-scroll');
        if (scroll) {
            scroll.addEventListener('click', function(e) {
                if (e.target === scroll && !state.isPopupMode) closePopup(5);
            });
        }

        // CTA button is now a tel: link — no redirect needed.
        // It stays in the popup; the user calls or closes manually.

        // Click on blurred image area highlights the form
        var gateImgWrap = document.getElementById('vizGateImgWrap');
        if (gateImgWrap) {
            gateImgWrap.addEventListener('click', function(e) {
                if (e.target.closest('.viz-popup-gate-form')) return;
                var formWrap = document.getElementById('vizGateFormWrap');
                if (formWrap) {
                    formWrap.classList.remove('viz-pulse');
                    formWrap.offsetHeight; // force reflow
                    formWrap.classList.add('viz-pulse');
                }
            });
        }
    }

    /* ============================================================
       SECTION 8: LEAD GATE (blur + timed tiers in popup)
       ============================================================ */
    var gateTimer20 = null;
    var gateTimer40 = null;

    function showLeadGate(imageUrl) {
        var gateImg = document.getElementById('vizGateImg');
        var gate = document.getElementById('vizPopupGate');
        var result = document.getElementById('vizPopupResult');

        // Show gate, hide result
        if (gate) { gate.style.display = ''; gate.style.opacity = ''; gate.style.transition = ''; }
        if (result) result.style.display = 'none';
        gateVisible = true;

        if (aceViz.settings.leadGate === 'blur') {
            var canvas = document.createElement('canvas');
            var ctx = canvas.getContext('2d');
            var img = new Image();
            img.onload = function() {
                canvas.width = img.width;
                canvas.height = img.height;
                ctx.filter = 'blur(20px)';
                ctx.drawImage(img, 0, 0);
                // Sharp corner (top-right 30%)
                ctx.filter = 'none';
                var rw = img.width * 0.3;
                var rh = img.height * 0.35;
                ctx.drawImage(img, img.width - rw, 0, rw, rh, img.width - rw, 0, rw, rh);
                gateImg.src = canvas.toDataURL('image/jpeg', 0.9);
                openPopup();
            };
            img.crossOrigin = 'anonymous';
            img.src = imageUrl;
        } else if (aceViz.settings.leadGate === 'watermark') {
            applyWatermark(imageUrl, gateImg);
            openPopup();
        }

        // Read phone from the rendered CTA
        var ctaPhoneEl = document.querySelector('.viz-cta-phone');
        var phoneHref = ctaPhoneEl ? ctaPhoneEl.getAttribute('href') : 'tel:3104386866';
        var phoneText = ctaPhoneEl ? ctaPhoneEl.textContent.trim() : 'or call: (310) 438-6866';

        // Tier 2: email alternative after 20s
        gateTimer20 = setTimeout(function() {
            var emailAlt = document.getElementById('vizEmailAlt');
            if (emailAlt && popupOpen && gateVisible) emailAlt.style.display = 'block';
        }, 20000);

    }

    function applyWatermark(imageUrl, targetImg) {
        var canvas = document.createElement('canvas');
        var ctx = canvas.getContext('2d');
        var img = new Image();
        img.onload = function() {
            canvas.width = img.width;
            canvas.height = img.height;
            ctx.drawImage(img, 0, 0);
            ctx.fillStyle = 'rgba(198,163,85,0.25)';
            ctx.font = 'bold 48px Inter, sans-serif';
            ctx.textAlign = 'center';
            ctx.fillText('ACE DESIGN & BUILD', canvas.width / 2, canvas.height / 2);
            ctx.font = '24px Inter, sans-serif';
            ctx.fillText('Book your free consultation', canvas.width / 2, canvas.height / 2 + 45);
            targetImg.src = canvas.toDataURL('image/jpeg', 0.9);
        };
        img.crossOrigin = 'anonymous';
        img.src = imageUrl;
    }

    function initLeadForm() {
        var form = document.getElementById('vizGateForm');
        if (!form) return;

        form.addEventListener('submit', function(e) {
            e.preventDefault();

            var name = document.getElementById('vizLeadName').value.trim();
            var phone = document.getElementById('vizLeadPhone').value.trim();
            var email = document.getElementById('vizLeadEmail').value.trim();
            var honeypot = document.getElementById('vizHoneypot');

            if (!name || !phone) return;
            // Don't check honeypot client-side - server handles it

            var btn = form.querySelector('.viz-btn-unlock');
            if (btn) {
                btn.disabled = true;
                btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Unlocking...';
            }

            var formData = new FormData();
            formData.append('action', 'ace_save_visualizer_lead');
            formData.append('nonce', aceViz.nonce);
            formData.append('name', name);
            formData.append('phone', phone);
            formData.append('email', email);
            formData.append('website_url', honeypot ? honeypot.value : '');
            formData.append('quiz_data', JSON.stringify(state.answers));
            formData.append('original_url', state.originalPreview || '');
            formData.append('generated_url', state.generatedImage || '');
            formData.append('price_low', state.priceRange ? state.priceRange.low : 0);
            formData.append('price_high', state.priceRange ? state.priceRange.high : 0);

            fetch(aceViz.ajax_url, { method: 'POST', body: formData }).then(function() {
                clearTimeout(gateTimer20);
                clearTimeout(gateTimer40);
                revealResult();
            }).catch(function() {
                clearTimeout(gateTimer20);
                clearTimeout(gateTimer40);
                revealResult();
            });
        });
    }

    /* ============================================================
       SECTION 9: RESULT + BEFORE/AFTER SLIDER
       Uses requestAnimationFrame + passive touch for 60fps
       ============================================================ */

    /** Exit popup mode to inline WITHOUT resetting state */
    function exitPopupToInline() {
        var flowPopup = document.getElementById('vizFlowPopup');
        var container = document.querySelector('.viz-container');
        var vizPopup = document.getElementById('vizPopup');
        var slot = document.getElementById('vizSlot');
        if (slot && container) slot.parentNode.insertBefore(container, slot.nextSibling);
        if (vizPopup) {
            vizPopup.classList.remove('flow-inline', 'visible');
            vizPopup.style.display = '';
            if (container && container.parentNode) container.parentNode.insertBefore(vizPopup, container.nextSibling);
        }
        if (flowPopup) { flowPopup.classList.remove('visible'); flowPopup.style.display = 'none'; }
        document.body.style.overflow = '';
        state.isPopupMode = false;
    }
    /** Open popup directly with result (no gate) */
    function showResult() {
        var beforeImg = document.getElementById('vizBefore');
        var afterImg = document.getElementById('vizAfter');
        var priceEl = document.getElementById('vizPrice');

        if (beforeImg) beforeImg.src = state.originalPreview;
        if (afterImg) afterImg.src = state.generatedImage;
        if (priceEl && state.priceRange) {
            priceEl.textContent = '$' + state.priceRange.low.toLocaleString() +
                ' - $' + state.priceRange.high.toLocaleString();
        }

        // Hide gate, show result
        var gate = document.getElementById('vizPopupGate');
        var result = document.getElementById('vizPopupResult');
        if (gate) gate.style.display = 'none';
        if (result) { result.style.display = ''; result.style.opacity = '1'; }

        openPopup();
        setTimeout(initBeforeAfterSlider, 500);
    }

    /** Smooth transition from gated view to result within the popup */
    function revealResult() {
        var gate = document.getElementById('vizPopupGate');
        var result = document.getElementById('vizPopupResult');

        // Set up result images
        var beforeImg = document.getElementById('vizBefore');
        var afterImg = document.getElementById('vizAfter');
        var priceEl = document.getElementById('vizPrice');

        if (beforeImg) beforeImg.src = state.originalPreview;
        if (afterImg) afterImg.src = state.generatedImage;
        if (priceEl && state.priceRange) {
            priceEl.textContent = '$' + state.priceRange.low.toLocaleString() +
                ' - $' + state.priceRange.high.toLocaleString();
        }

        gateVisible = false;

        // Fade out gate (1s)
        if (gate) {
            gate.style.transition = 'opacity 1s ease';
            gate.style.opacity = '0';
        }

        // After gate fades, swap to result view
        setTimeout(function() {
            if (gate) gate.style.display = 'none';
            if (result) {
                result.style.display = '';
                result.style.opacity = '0';
                result.offsetHeight; // force reflow
                result.style.transition = 'opacity 0.5s ease';
                result.style.opacity = '1';
            }

            // Scroll the appropriate container to top
            if (state.isPopupMode) {
                var flowInner = document.getElementById('vizFlowInner');
                if (flowInner) flowInner.scrollTop = 0;
            } else {
                var scroll = document.querySelector('.viz-popup-scroll');
                if (scroll) scroll.scrollTop = 0;
            }

            setTimeout(initBeforeAfterSlider, 500);
        }, 1000);
    }

    function initBeforeAfterSlider() {
        var container = document.getElementById('vizBA');
        var divider = document.getElementById('vizDivider');
        var beforeImg = document.getElementById('vizBefore');
        if (!container || !divider || !beforeImg) return;

        var isDragging = false;
        var pendingX = null;
        var rafId = null;

        // PERF: rAF-throttled — multiple mousemoves per frame only paint once
        function applyPosition() {
            if (pendingX === null) return;
            var rect = container.getBoundingClientRect();
            var pct = ((pendingX - rect.left) / rect.width) * 100;
            pct = Math.max(5, Math.min(95, pct));
            divider.style.left = pct + '%';
            beforeImg.style.clipPath = 'inset(0 ' + (100 - pct) + '% 0 0)';
            pendingX = null;
            rafId = null;
        }

        function setPosition(x) {
            pendingX = x;
            if (!rafId) rafId = requestAnimationFrame(applyPosition);
        }

        // Mouse — drag from anywhere on container
        container.addEventListener('mousedown', function(e) {
            e.preventDefault();
            isDragging = true;
            setPosition(e.clientX);
        });
        document.addEventListener('mousemove', function(e) {
            if (!isDragging) return;
            e.preventDefault();
            setPosition(e.clientX);
        });
        document.addEventListener('mouseup', function() {
            isDragging = false;
        });

        // Touch — drag from anywhere, prevent scroll while dragging
        container.addEventListener('touchstart', function(e) {
            isDragging = true;
            if (e.touches[0]) setPosition(e.touches[0].clientX);
        }, { passive: true });
        document.addEventListener('touchmove', function(e) {
            if (!isDragging) return;
            e.preventDefault();
            if (e.touches[0]) setPosition(e.touches[0].clientX);
        }, { passive: false });
        document.addEventListener('touchend', function() {
            isDragging = false;
        }, { passive: true });

        // Init at 50%
        var rect = container.getBoundingClientRect();
        setPosition(rect.left + container.offsetWidth / 2);
    }

    /* ============================================================
       SECTION 10: RETRY LOGIC
       ============================================================ */
    function initRetry() {
        var retryBtn = document.getElementById('vizRetry');
        if (!retryBtn) return;

        retryBtn.addEventListener('click', function() {
            state.retryCount++;

            if (state.retryCount > aceViz.settings.maxRetries) {
                retryBtn.textContent = 'For more design options, book your free consultation where our designer will create multiple custom concepts.';
                retryBtn.style.pointerEvents = 'none';
                retryBtn.style.color = '#9B978E';
                setTimeout(function() {
                    if (state.isPopupMode) {
                        closeVisualizerPopup();
                    } else {
                        closePopup(5);
                    }
                    setTimeout(function() {
                        var finalEl = document.getElementById('final');
                        if (finalEl) finalEl.scrollIntoView({ behavior: 'smooth' });
                    }, 400);
                }, 2000);
                return;
            }

            // Close result, go to step 2
            closePopup(2);

            // Reset feature selections for fresh choices
            setTimeout(function() {
                var selectedFeats = document.querySelectorAll('#vizStep3 .viz-option.selected');
                for (var i = 0; i < selectedFeats.length; i++) {
                    selectedFeats[i].classList.remove('selected');
                }
                state.answers.features = [];
                var cb = document.getElementById('vizFeatContinue');
                if (cb) cb.disabled = true;

                // Clear finish and style selections
                var selectedFinish = document.querySelectorAll('#vizStep4 .viz-option.selected');
                for (var j = 0; j < selectedFinish.length; j++) {
                    selectedFinish[j].classList.remove('selected');
                }
                var selectedStyle = document.querySelectorAll('#vizStep5 .viz-option.selected');
                for (var k = 0; k < selectedStyle.length; k++) {
                    selectedStyle[k].classList.remove('selected');
                }
                state.answers.finish = null;
                state.answers.style = null;

                // Re-highlight previous size
                if (state.answers.size) {
                    var sizeBtn = document.querySelector('#vizStep2 .viz-option[data-value="' + state.answers.size + '"]');
                    if (sizeBtn) sizeBtn.classList.add('selected');
                }
            }, 350);
        });
    }

    /* ============================================================
       SECTION 11: PROMO POPUP (scroll-triggered visualizer invite)
       ============================================================ */
    var promoShown = false;
    var promoDelayTimer = null;

    function initPromoPopup() {
        console.log('Promo: init called', 'promoPopup=' + aceViz.settings.promoPopup, 'el=' + !!document.getElementById('vizPromoPopup'));
        // Guard: setting disabled or popup HTML missing
        if (!aceViz.settings.promoPopup) { console.log('Promo: BLOCKED - setting disabled'); return; }
        if (!document.getElementById('vizPromoPopup')) { console.log('Promo: BLOCKED - HTML element missing'); return; }

        var threshold = aceViz.settings.promoScrollPct || 15;
        console.log('Promo: scroll handler attached, threshold=' + threshold + '%');

        window.addEventListener('scroll', onPromoScroll, { passive: true });

        function onPromoScroll() {
            if (promoShown) return;
            // Don't show promo if user has already used the visualizer
            if (state.hasUsedVisualizer) {
                promoShown = true;
                window.removeEventListener('scroll', onPromoScroll);
                return;
            }

            // Check scroll percentage first
            var scrollTop = window.pageYOffset || document.documentElement.scrollTop;
            var docHeight = document.documentElement.scrollHeight - window.innerHeight;
            if (docHeight <= 0) return;
            var scrollPct = (scrollTop / docHeight) * 100;
            console.log('Promo: scroll ' + scrollPct.toFixed(1) + '%');

            if (scrollPct >= threshold) {
                console.log('Promo: threshold reached at ' + scrollPct.toFixed(1) + '%');
                promoShown = true;
                window.removeEventListener('scroll', onPromoScroll);

                // Delay 1 second after reaching threshold
                console.log('Promo: will show popup in 1s');
                promoDelayTimer = setTimeout(openPromoPopup, 1000);
            }
        }
    }

    function openPromoPopup() {
        var popup = document.getElementById('vizPromoPopup');
        if (!popup) { console.log('Promo: BLOCKED in open - element missing'); return; }
        // Don't open if result popup or flow popup is already showing
        if (popupOpen || state.isPopupMode) { console.log('Promo: BLOCKED in open - another popup is open'); return; }
        console.log('Promo: showing popup NOW');
        popup.style.display = 'block';
        popup.offsetHeight; // force reflow
        popup.classList.add('visible');
        document.body.style.overflow = 'hidden';
    }

    function closePromoPopup() {
        var popup = document.getElementById('vizPromoPopup');
        if (!popup) return;
        clearTimeout(promoDelayTimer);

        // Closing animation
        popup.classList.remove('visible');
        popup.classList.add('closing');

        setTimeout(function() {
            popup.style.display = 'none';
            popup.classList.remove('closing');
            document.body.style.overflow = '';
        }, 200);
    }

    function initPromoListeners() {
        // Close X
        var closeBtn = document.getElementById('vizPromoClose');
        if (closeBtn) closeBtn.addEventListener('click', closePromoPopup);

        // Dismiss link
        var dismiss = document.getElementById('vizPromoDismiss');
        if (dismiss) dismiss.addEventListener('click', closePromoPopup);

        // Backdrop click
        var backdrop = document.getElementById('vizPromoBackdrop');
        if (backdrop) backdrop.addEventListener('click', closePromoPopup);

        // ESC handled in initPopup (consolidated handler)

        // CTA button — close promo, open visualizer flow popup
        var cta = document.getElementById('vizPromoCta');
        if (cta) {
            cta.addEventListener('click', function() {
                closePromoPopup();
                setTimeout(function() {
                    openVisualizerPopup();
                }, 250);
            });
        }
    }

    /* ============================================================
       SECTION 12: VISUALIZER FLOW POPUP (full wizard in modal)
       Moves .viz-container into a popup overlay for popup mode.
       ============================================================ */

    /** Reset the entire visualizer UI + state to fresh Step 1 */
    function resetVisualizerState() {
        // Invalidate any in-flight generation
        currentGenId++;
        stopLoading();
        clearTimeout(gateTimer20);
        clearTimeout(gateTimer40);

        // Reset state
        state.currentStep = 1;
        state.imageFile = null;
        state.imageOrientation = null;
        state.answers = { size: null, features: [], finish: null, style: null };
        state.generatedImage = null;
        state.originalPreview = null;
        state.priceRange = null;
        // Note: don't reset retryCount (persists across sessions)

        popupOpen = false;
        gateVisible = false;

        // Reset upload UI
        var uploadContent = document.getElementById('vizUploadContent');
        var preview = document.getElementById('vizPreview');
        var success = document.getElementById('vizUploadSuccess');
        var fileInput = document.getElementById('vizFileInput');
        if (uploadContent) uploadContent.style.display = '';
        if (preview) preview.style.display = 'none';
        if (success) success.style.display = 'none';
        if (fileInput) fileInput.value = '';
        hideError();

        // Clear all quiz selections
        var allSelected = document.querySelectorAll('.viz-option.selected');
        for (var i = 0; i < allSelected.length; i++) {
            allSelected[i].classList.remove('selected');
        }
        var cb = document.getElementById('vizFeatContinue');
        if (cb) cb.disabled = true;

        // Reset all steps to inactive, activate Step 1
        var allSteps = document.querySelectorAll('.viz-step');
        for (var n = 0; n < allSteps.length; n++) {
            allSteps[n].classList.remove('active', 'exit-left');
        }
        var step1 = document.getElementById('vizStep1');
        if (step1) step1.classList.add('active');
        updateProgress(1);

        // Reset result popup views
        resetPopupViews();

        // Ensure viz-container is visible
        var container = document.querySelector('.viz-container');
        if (container) {
            container.style.display = '';
            container.classList.remove('viz-showing-result');
        }

        // Hide #vizPopup
        var vizPopup = document.getElementById('vizPopup');
        if (vizPopup) {
            vizPopup.classList.remove('flow-inline', 'visible');
            vizPopup.style.display = 'none';
        }

        // Remove any lingering generation error
        var existingErr = document.querySelector('#vizStep5 .viz-upload-error');
        if (existingErr) existingErr.remove();
    }

    function openVisualizerPopup() {
        if (state.isPopupMode) return; // already open

        var flowPopup = document.getElementById('vizFlowPopup');
        var flowBody = document.getElementById('vizFlowBody');
        var container = document.querySelector('.viz-container');
        var vizPopup = document.getElementById('vizPopup');
        if (!flowPopup || !flowBody || !container) return;

        // Mark that user has engaged with visualizer (suppresses promo popup)
        state.hasUsedVisualizer = true;

        // Set popup mode first (so resetVisualizerState scroll logic works)
        state.isPopupMode = true;

        // Reset everything to fresh Step 1
        resetVisualizerState();

        // Move viz-container into flow popup
        flowBody.appendChild(container);

        // Move result popup into flow popup (rendered inline via flow-inline class)
        if (vizPopup) {
            vizPopup.style.display = ''; // clear any inline display
            vizPopup.classList.add('flow-inline');
            flowBody.appendChild(vizPopup);
        }

        // Show flow popup
        flowPopup.style.display = 'block';
        flowPopup.offsetHeight; // force reflow
        flowPopup.classList.add('visible');
        document.body.style.overflow = 'hidden';
    }

    function closeVisualizerPopup() {
        if (!state.isPopupMode) return;

        var flowPopup = document.getElementById('vizFlowPopup');
        if (!flowPopup) return;

        // Clean up any active result/gate state
        popupOpen = false;
        gateVisible = false;
        currentGenId++; // invalidate in-flight generation
        stopLoading();
        clearTimeout(gateTimer20);
        clearTimeout(gateTimer40);

        // Fade out
        flowPopup.classList.remove('visible');

        setTimeout(function() {
            flowPopup.style.display = 'none';
            document.body.style.overflow = '';

            // Move elements back to original section
            var container = document.querySelector('.viz-container');
            var vizPopup = document.getElementById('vizPopup');
            var slot = document.getElementById('vizSlot');

            if (slot && container) {
                slot.parentNode.insertBefore(container, slot.nextSibling);
            }
            if (vizPopup) {
                vizPopup.classList.remove('flow-inline', 'visible');
                vizPopup.style.display = ''; // let CSS handle default (display: none)
                if (container && container.parentNode) {
                    container.parentNode.insertBefore(vizPopup, container.nextSibling);
                }
            }

            // Reset mode and state
            state.isPopupMode = false;
            resetVisualizerState();
        }, 300);
    }

    // Expose globally for hero CTA onclick and nav link close
    window.openVisualizerPopup = openVisualizerPopup;
    window.closeVisualizerPopup = closeVisualizerPopup;

    function initFlowPopup() {
        // Close button
        var closeBtn = document.getElementById('vizFlowClose');
        if (closeBtn) {
            closeBtn.addEventListener('click', function() {
                closeVisualizerPopup();
            });
        }

        // Backdrop click (clicking empty area of scroll wrapper)
        var scroll = document.getElementById('vizFlowScroll');
        if (scroll) {
            scroll.addEventListener('click', function(e) {
                if (e.target === scroll) closeVisualizerPopup();
            });
        }

        // Hero CTA button
        var heroBtn = document.getElementById('heroVizBtn');
        if (heroBtn) {
            heroBtn.addEventListener('click', function(e) {
                e.preventDefault();
                openVisualizerPopup();
            });
        }
    }

    /* ============================================================
       SECTION 13: INIT
       ============================================================ */
    document.addEventListener('DOMContentLoaded', function() {
        console.log('visualizer.js DOMContentLoaded fired, vizStep1=' + !!document.getElementById('vizStep1'));
        // Only init if visualizer HTML is present
        if (!document.getElementById('vizStep1')) return;
        initUpload();
        initQuiz();
        initPopup();
        initLeadForm();
        initRetry();
        initFlowPopup();
        initPromoListeners();
        initPromoPopup();
        var step1 = document.getElementById('vizStep1');
        if (step1 && step1.classList.contains('active')) {
            updateProgress(1);
        } else {
            goToStep(1);
        }
    });

})();
