<?php
/**
 * Plugin Name: ACE Theme Manager
 * Plugin URI:  https://acedesignbuild.com
 * Description: Complete content management plugin for the ACE Design & Build WordPress theme. Manages all site content, leads, tracking, and SEO.
 * Version:     1.0.0
 * Author:      ACE Design & Build
 * Author URI:  https://acedesignbuild.com
 * License:     GPL-2.0-or-later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: ace-theme-manager
 * Domain Path: /languages
 * Requires at least: 6.4
 * Requires PHP: 8.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Plugin constants.
define( 'ACE_PLUGIN_VERSION', '1.0.0' );
define( 'ACE_PLUGIN_PATH', plugin_dir_path( __FILE__ ) );
define( 'ACE_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

/**
 * Plugin activation: create leads table and set default options.
 */
function ace_plugin_activate() {
	if ( ! current_user_can( 'activate_plugins' ) ) {
		return;
	}

	// Create leads table.
	global $wpdb;
	$table_name      = $wpdb->prefix . 'ace_leads';
	$charset_collate = $wpdb->get_charset_collate();

	$sql = "CREATE TABLE {$table_name} (
		id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
		full_name VARCHAR(255) NOT NULL,
		phone VARCHAR(50) NOT NULL,
		email VARCHAR(255) NOT NULL,
		service VARCHAR(255),
		message TEXT,
		preferred_contact VARCHAR(50),
		source_page VARCHAR(255),
		source_url VARCHAR(500),
		ip_address VARCHAR(45),
		user_agent TEXT,
		created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
		is_read TINYINT(1) DEFAULT 0,
		notes TEXT,
		PRIMARY KEY  (id)
	) {$charset_collate};";

	require_once ABSPATH . 'wp-admin/includes/upgrade.php';
	dbDelta( $sql );

	// Set default options.
	require_once ACE_PLUGIN_PATH . 'includes/default-content.php';
	if ( function_exists( 'ace_get_default_options' ) ) {
		$defaults = ace_get_default_options();
		foreach ( $defaults as $key => $value ) {
			if ( false === get_option( $key ) ) {
				add_option( $key, $value );
			}
		}
	}

	// Install default options for form/tracking if not present.
	if ( false === get_option( 'ace_form_recipients' ) ) {
		add_option( 'ace_form_recipients', 'info@acedesignbuild.com' );
	}
	if ( false === get_option( 'ace_form_success_message' ) ) {
		add_option( 'ace_form_success_message', "Thank you! Your request has been submitted. We'll be in touch within 2 hours during business hours." );
	}
	if ( false === get_option( 'ace_form_thank_you_url' ) ) {
		add_option( 'ace_form_thank_you_url', '/thank-you/' );
	}

	flush_rewrite_rules();
}
register_activation_hook( __FILE__, 'ace_plugin_activate' );

/**
 * Plugin deactivation.
 */
function ace_plugin_deactivate() {
	flush_rewrite_rules();
}
register_deactivation_hook( __FILE__, 'ace_plugin_deactivate' );

/**
 * Load plugin classes and initialize.
 */
function ace_plugin_init() {
	// Load translation files.
	load_plugin_textdomain( 'ace-theme-manager', false, dirname( plugin_basename( __FILE__ ) ) . '/languages' );

	// Load includes.
	require_once ACE_PLUGIN_PATH . 'includes/default-content.php';
	require_once ACE_PLUGIN_PATH . 'includes/class-settings.php';
	require_once ACE_PLUGIN_PATH . 'includes/class-ace-lead-form.php';
	require_once ACE_PLUGIN_PATH . 'includes/class-tracking.php';
	require_once ACE_PLUGIN_PATH . 'includes/class-import-export.php';

	// Initialize lead form (AJAX handlers).
	if ( class_exists( 'ACE_Lead_Form' ) ) {
		$lead_form = new ACE_Lead_Form();
		$lead_form->init();
	}

	// Initialize tracking (front-end scripts).
	if ( ! is_admin() && class_exists( 'ACE_Tracking' ) ) {
		$tracking = new ACE_Tracking();
		$tracking->init();
	}

	// Initialize import/export.
	if ( class_exists( 'ACE_Import_Export' ) ) {
		new ACE_Import_Export();
	}
}
add_action( 'plugins_loaded', 'ace_plugin_init' );

/**
 * Load admin-only classes.
 */
function ace_plugin_admin_init() {
	if ( ! is_admin() ) {
		return;
	}

	require_once ACE_PLUGIN_PATH . 'includes/class-admin-page.php';
	require_once ACE_PLUGIN_PATH . 'includes/class-lead-table.php';

	if ( class_exists( 'ACE_Admin_Page' ) ) {
		new ACE_Admin_Page();
	}
}
add_action( 'init', 'ace_plugin_admin_init' );
