<?php
/**
 * ACE Settings helper class.
 *
 * Provides a centralised, static API for reading and writing plugin options
 * stored in the WordPress options table. Every option managed by the ACE
 * Theme Manager plugin uses the `ace_` prefix unless accessed through the
 * raw getter.
 *
 * @package ACE_Theme_Manager
 * @since   1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class ACE_Settings
 *
 * Static helper for ACE Theme Manager option retrieval and persistence.
 *
 * @since 1.0.0
 */
class ACE_Settings {

	/**
	 * Option name prefix applied to all managed keys.
	 *
	 * @var string
	 */
	const PREFIX = 'ace_';

	/**
	 * Retrieve a prefixed plugin option.
	 *
	 * Looks up `ace_{$key}` in the options table. If the stored value is
	 * empty (empty string, null, or false) the supplied default is returned
	 * instead.
	 *
	 * @since 1.0.0
	 *
	 * @param string $key     Option key without the `ace_` prefix.
	 * @param mixed  $default Optional. Fallback value. Default empty string.
	 * @return mixed Sanitized option value or the default.
	 */
	public static function get( $key, $default = '' ) {
		$key   = sanitize_key( $key );
		$value = get_option( self::PREFIX . $key );

		if ( empty( $value ) ) {
			return $default;
		}

		return $value;
	}

	/**
	 * Retrieve a raw option value without the plugin prefix.
	 *
	 * Useful when reading options that are not prefixed with `ace_` or when
	 * the full option name is already known.
	 *
	 * @since 1.0.0
	 *
	 * @param string $key Full option name.
	 * @return mixed Option value or false if the option does not exist.
	 */
	public static function get_raw( $key ) {
		$key = sanitize_key( $key );

		return get_option( $key );
	}

	/**
	 * Retrieve a single service entry by its slug.
	 *
	 * Searches the `ace_services` option array for a service whose `slug`
	 * field matches the provided value.
	 *
	 * @since 1.0.0
	 *
	 * @param string $slug The service slug to look up.
	 * @return array|false The service data array on success, false on failure.
	 */
	public static function get_service( $slug ) {
		$slug     = sanitize_title( $slug );
		$services = get_option( 'ace_services', array() );

		if ( ! is_array( $services ) || empty( $services ) ) {
			return false;
		}

		foreach ( $services as $service ) {
			if ( is_array( $service ) && isset( $service['slug'] ) && sanitize_title( $service['slug'] ) === $slug ) {
				return $service;
			}
		}

		return false;
	}

	/**
	 * Retrieve the full services array.
	 *
	 * @since 1.0.0
	 *
	 * @return array List of service arrays. Empty array when nothing is stored.
	 */
	public static function get_all_services() {
		$services = get_option( 'ace_services', array() );

		if ( ! is_array( $services ) ) {
			return array();
		}

		return $services;
	}

	/**
	 * Retrieve the testimonials array.
	 *
	 * @since 1.0.0
	 *
	 * @return array List of testimonial arrays. Empty array when nothing is stored.
	 */
	public static function get_testimonials() {
		$testimonials = get_option( 'ace_testimonials', array() );

		if ( ! is_array( $testimonials ) ) {
			return array();
		}

		return $testimonials;
	}

	/**
	 * Retrieve the SEO meta data for a specific page.
	 *
	 * Returns the associative array of SEO fields (title, description, etc.)
	 * for the given page key, or an empty array when no data exists.
	 *
	 * @since 1.0.0
	 *
	 * @param string $page_key The page identifier (e.g. "home", "about").
	 * @return array Associative array of SEO fields. Empty array on failure.
	 */
	public static function get_seo( $page_key ) {
		$page_key = sanitize_key( $page_key );
		$seo      = get_option( 'ace_seo', array() );

		if ( ! is_array( $seo ) || ! isset( $seo[ $page_key ] ) || ! is_array( $seo[ $page_key ] ) ) {
			return array();
		}

		return $seo[ $page_key ];
	}

	/**
	 * Save a prefixed plugin option.
	 *
	 * Persists the value under `ace_{$key}` using `update_option`, which
	 * creates the option automatically if it does not already exist.
	 *
	 * @since 1.0.0
	 *
	 * @param string $key   Option key without the `ace_` prefix.
	 * @param mixed  $value The value to store.
	 * @return bool True if the option was updated, false otherwise.
	 */
	public static function save( $key, $value ) {
		$key = sanitize_key( $key );

		return update_option( self::PREFIX . $key, $value );
	}

	/**
	 * Retrieve the site statistics option.
	 *
	 * @since 1.0.0
	 *
	 * @return array Associative array of stat entries. Empty array when nothing is stored.
	 */
	public static function get_stats() {
		$stats = get_option( 'ace_stats', array() );

		if ( ! is_array( $stats ) ) {
			return array();
		}

		return $stats;
	}

	/**
	 * Retrieve the process steps option.
	 *
	 * @since 1.0.0
	 *
	 * @return array List of process step arrays. Empty array when nothing is stored.
	 */
	public static function get_process_steps() {
		$steps = get_option( 'ace_process_steps', array() );

		if ( ! is_array( $steps ) ) {
			return array();
		}

		return $steps;
	}

	/**
	 * Retrieve the "Why ACE" features option.
	 *
	 * @since 1.0.0
	 *
	 * @return array List of feature arrays. Empty array when nothing is stored.
	 */
	public static function get_why_ace_features() {
		$features = get_option( 'ace_why_ace_features', array() );

		if ( ! is_array( $features ) ) {
			return array();
		}

		return $features;
	}

	/**
	 * Retrieve the about page values option.
	 *
	 * @since 1.0.0
	 *
	 * @return array List of value arrays. Empty array when nothing is stored.
	 */
	public static function get_about_values() {
		$values = get_option( 'ace_about_values', array() );

		if ( ! is_array( $values ) ) {
			return array();
		}

		return $values;
	}

	/**
	 * Retrieve the service areas option.
	 *
	 * @since 1.0.0
	 *
	 * @return array List of service area entries. Empty array when nothing is stored.
	 */
	public static function get_service_areas() {
		$areas = get_option( 'ace_service_areas', array() );

		if ( ! is_array( $areas ) ) {
			return array();
		}

		return $areas;
	}

	/**
	 * Check whether a front-end section is visible.
	 *
	 * Reads the `ace_sections` option (expected to be an associative array
	 * of section keys mapped to boolean-like values) and determines whether
	 * the requested section should be rendered.
	 *
	 * @since 1.0.0
	 *
	 * @param string $section_key The section identifier to check.
	 * @return bool True if the section is visible, false otherwise.
	 */
	public static function is_section_visible( $section_key ) {
		$section_key = sanitize_key( $section_key );
		$sections    = get_option( 'ace_sections', array() );

		if ( ! is_array( $sections ) || ! isset( $sections[ $section_key ] ) ) {
			return false;
		}

		return (bool) $sections[ $section_key ];
	}
}
