<?php
/**
 * Tracking Scripts Handler.
 *
 * Outputs analytics and conversion tracking scripts (GA4, GTM, Facebook
 * Pixel, Google Ads, and custom scripts) to the front-end. Also provides
 * a static helper for firing conversion events after form submissions.
 *
 * @package ACE_Theme_Manager
 * @since   1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class ACE_Tracking
 *
 * Manages the injection of third-party tracking snippets into the
 * front-end `<head>`, after `<body>`, and before `</body>` via the
 * standard WordPress hook points.
 *
 * @since 1.0.0
 */
class ACE_Tracking {

	/**
	 * Initialize the tracking hooks.
	 *
	 * Registers output callbacks on `wp_head`, `wp_body_open`, and
	 * `wp_footer` so that tracking scripts are rendered at the correct
	 * positions in the document. Only activates on the front-end.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function init() {
		// Only output scripts on the public-facing front-end.
		if ( is_admin() ) {
			return;
		}

		add_action( 'wp_head', array( $this, 'output_head_scripts' ), 1 );
		add_action( 'wp_body_open', array( $this, 'output_body_scripts' ), 1 );
		add_action( 'wp_footer', array( $this, 'output_footer_scripts' ), 99 );
	}

	/**
	 * Output tracking scripts in the document `<head>`.
	 *
	 * Renders GA4 (gtag.js), GTM head snippet, Facebook Pixel base code,
	 * and any custom head scripts stored in plugin options.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function output_head_scripts() {
		$this->output_ga4_script();
		$this->output_gtm_head_script();
		$this->output_facebook_pixel_script();
		$this->output_custom_scripts( 'ace_custom_head_scripts' );
	}

	/**
	 * Output tracking scripts immediately after the opening `<body>` tag.
	 *
	 * Renders the GTM noscript fallback iframe and any custom body-open
	 * scripts stored in plugin options.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function output_body_scripts() {
		$this->output_gtm_body_noscript();
		$this->output_custom_scripts( 'ace_custom_body_scripts' );
	}

	/**
	 * Output tracking scripts before the closing `</body>` tag.
	 *
	 * Renders any custom footer scripts stored in plugin options.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function output_footer_scripts() {
		$this->output_custom_scripts( 'ace_custom_footer_scripts' );
	}

	/**
	 * Output the Google Analytics 4 (gtag.js) snippet.
	 *
	 * Loads the gtag.js library asynchronously and initialises the GA4
	 * measurement stream identified by the `ace_ga4_id` option.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	private function output_ga4_script() {
		$ga4_id = get_option( 'ace_ga4_id', '' );

		if ( empty( $ga4_id ) ) {
			return;
		}

		$ga4_id = sanitize_text_field( $ga4_id );
		?>
<!-- Google Analytics 4 (gtag.js) - ACE Theme Manager -->
<script async src="https://www.googletagmanager.com/gtag/js?id=<?php echo esc_attr( $ga4_id ); ?>"></script>
<script>
window.dataLayer = window.dataLayer || [];
function gtag(){dataLayer.push(arguments);}
gtag('js', new Date());
gtag('config', '<?php echo esc_js( $ga4_id ); ?>');
</script>
		<?php
	}

	/**
	 * Output the Google Tag Manager head snippet.
	 *
	 * Injects the GTM container script into `<head>` using the container
	 * ID stored in the `ace_gtm_id` option.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	private function output_gtm_head_script() {
		$gtm_id = get_option( 'ace_gtm_id', '' );

		if ( empty( $gtm_id ) ) {
			return;
		}

		$gtm_id = sanitize_text_field( $gtm_id );
		?>
<!-- Google Tag Manager - ACE Theme Manager -->
<script>
(function(w,d,s,l,i){w[l]=w[l]||[];w[l].push({'gtm.start':
new Date().getTime(),event:'gtm.js'});var f=d.getElementsByTagName(s)[0],
j=d.createElement(s),dl=l!='dataLayer'?'&l='+l:'';j.async=true;j.src=
'https://www.googletagmanager.com/gtm.js?id='+i+dl;f.parentNode.insertBefore(j,f);
})(window,document,'script','dataLayer','<?php echo esc_js( $gtm_id ); ?>');
</script>
<!-- End Google Tag Manager -->
		<?php
	}

	/**
	 * Output the Google Tag Manager noscript fallback.
	 *
	 * Renders a noscript iframe immediately after `<body>` so that GTM
	 * can still function for users with JavaScript disabled.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	private function output_gtm_body_noscript() {
		$gtm_id = get_option( 'ace_gtm_id', '' );

		if ( empty( $gtm_id ) ) {
			return;
		}

		$gtm_id = sanitize_text_field( $gtm_id );
		?>
<!-- Google Tag Manager (noscript) - ACE Theme Manager -->
<noscript><iframe src="https://www.googletagmanager.com/ns.html?id=<?php echo esc_attr( $gtm_id ); ?>"
height="0" width="0" style="display:none;visibility:hidden"></iframe></noscript>
<!-- End Google Tag Manager (noscript) -->
		<?php
	}

	/**
	 * Output the Facebook (Meta) Pixel base code.
	 *
	 * Initialises the Facebook Pixel using the pixel ID stored in the
	 * `ace_fb_pixel_id` option and fires a default PageView event.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	private function output_facebook_pixel_script() {
		$pixel_id = get_option( 'ace_fb_pixel_id', '' );

		if ( empty( $pixel_id ) ) {
			return;
		}

		$pixel_id = sanitize_text_field( $pixel_id );
		?>
<!-- Facebook Pixel - ACE Theme Manager -->
<script>
!function(f,b,e,v,n,t,s)
{if(f.fbq)return;n=f.fbq=function(){n.callMethod?
n.callMethod.apply(n,arguments):n.queue.push(arguments)};
if(!f._fbq)f._fbq=n;n.push=n;n.loaded=!0;n.version='2.0';
n.queue=[];t=b.createElement(e);t.async=!0;
t.src=v;s=b.getElementsByTagName(e)[0];
s.parentNode.insertBefore(t,s)}(window, document,'script',
'https://connect.facebook.net/en_US/fbevents.js');
fbq('init', '<?php echo esc_js( $pixel_id ); ?>');
fbq('track', 'PageView');
</script>
<noscript><img height="1" width="1" style="display:none"
src="https://www.facebook.com/tr?id=<?php echo esc_attr( $pixel_id ); ?>&ev=PageView&noscript=1"
/></noscript>
<!-- End Facebook Pixel -->
		<?php
	}

	/**
	 * Output custom script content from a plugin option.
	 *
	 * Retrieves the raw script markup stored in the given option key and
	 * outputs it without escaping, as it is expected to contain valid
	 * `<script>` tags or other tracking markup entered by the site admin.
	 *
	 * @since 1.0.0
	 *
	 * @param string $option_key The option name containing the raw script content.
	 * @return void
	 */
	private function output_custom_scripts( $option_key ) {
		$scripts = get_option( $option_key, '' );

		if ( empty( $scripts ) ) {
			return;
		}

		// Output raw - site administrators are trusted to enter valid markup.
		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		echo "\n" . $scripts . "\n";
	}

	/**
	 * Generate conversion tracking script markup after a form submission.
	 *
	 * Returns a string of JavaScript that fires conversion events on GA4,
	 * Facebook Pixel, and Google Ads when called. Intended to be embedded
	 * in the thank-you page or returned via AJAX after a successful lead
	 * capture submission.
	 *
	 * @since 1.0.0
	 *
	 * @param array $lead_data {
	 *     Optional. Lead details to include in conversion payloads.
	 *
	 *     @type string $email   Lead email address.
	 *     @type string $phone   Lead phone number.
	 *     @type string $service Service type requested.
	 * }
	 * @return string HTML script block with conversion event calls, or empty string.
	 */
	public static function output_conversion_scripts( $lead_data = array() ) {
		$ga4_id         = get_option( 'ace_ga4_id', '' );
		$fb_pixel_id    = get_option( 'ace_fb_pixel_id', '' );
		$gads_id        = get_option( 'ace_gads_conversion_id', '' );
		$gads_label     = get_option( 'ace_gads_conversion_label', '' );

		// Nothing to output if no tracking IDs are configured.
		if ( empty( $ga4_id ) && empty( $fb_pixel_id ) && empty( $gads_id ) ) {
			return '';
		}

		$service = isset( $lead_data['service'] ) ? sanitize_text_field( $lead_data['service'] ) : '';
		$email   = isset( $lead_data['email'] ) ? sanitize_email( $lead_data['email'] ) : '';
		$phone   = isset( $lead_data['phone'] ) ? sanitize_text_field( $lead_data['phone'] ) : '';

		$scripts = "<script>\n";

		// GA4 generate_lead event.
		if ( ! empty( $ga4_id ) ) {
			$scripts .= "/* GA4 Conversion */\n";
			$scripts .= "if (typeof gtag === 'function') {\n";
			$scripts .= "\tgtag('event', 'generate_lead', {\n";
			$scripts .= "\t\t'event_category': 'Lead Form',\n";
			$scripts .= "\t\t'event_label': " . wp_json_encode( $service ) . ",\n";
			$scripts .= "\t\t'value': 1\n";
			$scripts .= "\t});\n";
			$scripts .= "}\n";
		}

		// Facebook Pixel Lead event.
		if ( ! empty( $fb_pixel_id ) ) {
			$scripts .= "/* Facebook Pixel Conversion */\n";
			$scripts .= "if (typeof fbq === 'function') {\n";
			$scripts .= "\tfbq('track', 'Lead', {\n";
			$scripts .= "\t\t'content_name': " . wp_json_encode( $service ) . ",\n";
			$scripts .= "\t\t'content_category': 'Lead Form'\n";
			$scripts .= "\t});\n";
			$scripts .= "}\n";
		}

		// Google Ads conversion event.
		if ( ! empty( $gads_id ) && ! empty( $gads_label ) ) {
			$send_to = esc_js( sanitize_text_field( $gads_id ) ) . '/' . esc_js( sanitize_text_field( $gads_label ) );
			$scripts .= "/* Google Ads Conversion */\n";
			$scripts .= "if (typeof gtag === 'function') {\n";
			$scripts .= "\tgtag('event', 'conversion', {\n";
			$scripts .= "\t\t'send_to': '" . $send_to . "'\n";
			$scripts .= "\t});\n";
			$scripts .= "}\n";
		}

		$scripts .= "</script>\n";

		return $scripts;
	}
}
