/**
 * ACE Design & Build - Lead Form Handler
 * AJAX form submission with validation
 * @version 1.0.0
 */

(function() {
    'use strict';

    document.addEventListener('DOMContentLoaded', function() {
        var forms = document.querySelectorAll('.ace-lead-form');
        forms.forEach(function(form) {
            initForm(form);
        });
    });

    function initForm(form) {
        form.addEventListener('submit', function(e) {
            e.preventDefault();

            // Clear previous errors
            clearErrors(form);

            // Validate
            if (!validateForm(form)) {
                return;
            }

            // Show loading state
            setLoadingState(form, true);

            // Collect form data
            var formData = new FormData(form);

            // Send AJAX request
            var xhr = new XMLHttpRequest();
            xhr.open('POST', aceForm.ajaxUrl, true);

            xhr.onload = function() {
                setLoadingState(form, false);

                try {
                    var response = JSON.parse(xhr.responseText);

                    if (response.success) {
                        showMessage(form, response.data.message || 'Thank you! Your request has been submitted.', 'success');
                        form.reset();

                        // Fire conversion tracking
                        fireConversionEvents(formData);

                        // Redirect to thank you page if configured
                        if (response.data.redirect) {
                            setTimeout(function() {
                                window.location.href = response.data.redirect;
                            }, 1500);
                        }
                    } else {
                        // Handle validation errors
                        if (response.data && response.data.errors) {
                            var errors = response.data.errors;
                            for (var field in errors) {
                                if (errors.hasOwnProperty(field)) {
                                    showFieldError(form, field, errors[field]);
                                }
                            }
                        }
                        showMessage(form, response.data.message || 'Something went wrong. Please try again.', 'error');
                    }
                } catch(parseError) {
                    showMessage(form, 'Something went wrong. Please try again or call us at ' + getPhoneNumber() + '.', 'error');
                }
            };

            xhr.onerror = function() {
                setLoadingState(form, false);
                showMessage(form, 'Connection error. Please try again or call us at ' + getPhoneNumber() + '.', 'error');
            };

            xhr.send(formData);
        });

        // Real-time validation on blur
        var inputs = form.querySelectorAll('input[required], select[required]');
        inputs.forEach(function(input) {
            input.addEventListener('blur', function() {
                validateField(this);
            });

            // Clear error on input
            input.addEventListener('input', function() {
                var errorEl = this.parentElement.querySelector('.field-error');
                if (errorEl) errorEl.textContent = '';
                this.classList.remove('error');
            });
        });
    }

    function validateForm(form) {
        var isValid = true;

        // Full Name
        var name = form.querySelector('[name="full_name"]');
        if (name && !name.value.trim()) {
            showFieldError(form, 'full_name', 'This field is required.');
            isValid = false;
        }

        // Phone
        var phone = form.querySelector('[name="phone"]');
        if (phone) {
            var phoneVal = phone.value.trim();
            if (!phoneVal) {
                showFieldError(form, 'phone', 'This field is required.');
                isValid = false;
            } else if (!isValidPhone(phoneVal)) {
                showFieldError(form, 'phone', 'Please enter a valid phone number.');
                isValid = false;
            }
        }

        // Email
        var email = form.querySelector('[name="email"]');
        if (email) {
            var emailVal = email.value.trim();
            if (!emailVal) {
                showFieldError(form, 'email', 'This field is required.');
                isValid = false;
            } else if (!isValidEmail(emailVal)) {
                showFieldError(form, 'email', 'Please enter a valid email address.');
                isValid = false;
            }
        }

        return isValid;
    }

    function validateField(input) {
        var name = input.getAttribute('name');
        var value = input.value.trim();
        var form = input.closest('form');

        if (input.hasAttribute('required') && !value) {
            showFieldError(form, name, 'This field is required.');
            return false;
        }

        if (name === 'email' && value && !isValidEmail(value)) {
            showFieldError(form, name, 'Please enter a valid email address.');
            return false;
        }

        if (name === 'phone' && value && !isValidPhone(value)) {
            showFieldError(form, name, 'Please enter a valid phone number.');
            return false;
        }

        return true;
    }

    function showFieldError(form, fieldName, message) {
        var field = form.querySelector('[name="' + fieldName + '"]');
        if (!field) return;

        field.classList.add('error');
        var errorEl = field.parentElement.querySelector('.field-error');
        if (errorEl) {
            errorEl.textContent = message;
        }
    }

    function clearErrors(form) {
        var errors = form.querySelectorAll('.field-error');
        errors.forEach(function(el) { el.textContent = ''; });

        var fields = form.querySelectorAll('.error');
        fields.forEach(function(el) { el.classList.remove('error'); });

        var msg = form.querySelector('.form-message');
        if (msg) msg.style.display = 'none';
    }

    function showMessage(form, message, type) {
        var msgEl = form.querySelector('.form-message');
        if (!msgEl) return;

        msgEl.textContent = message;
        msgEl.className = 'form-message ' + type;
        msgEl.style.display = 'block';

        // Scroll to message
        msgEl.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
    }

    function setLoadingState(form, loading) {
        var submitBtn = form.querySelector('[type="submit"]');
        if (!submitBtn) return;

        var btnText = submitBtn.querySelector('.btn-text');
        var btnLoading = submitBtn.querySelector('.btn-loading');

        if (loading) {
            submitBtn.disabled = true;
            if (btnText) btnText.style.display = 'none';
            if (btnLoading) btnLoading.style.display = 'inline-flex';
        } else {
            submitBtn.disabled = false;
            if (btnText) btnText.style.display = '';
            if (btnLoading) btnLoading.style.display = 'none';
        }
    }

    function isValidEmail(email) {
        return /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email);
    }

    function isValidPhone(phone) {
        // Accept various phone formats: (310) 438-6866, 310-438-6866, 3104386866, +13104386866, etc.
        var cleaned = phone.replace(/[\s\-\(\)\.\+]/g, '');
        return /^\d{10,15}$/.test(cleaned);
    }

    function getPhoneNumber() {
        var phoneLink = document.querySelector('.header-phone span');
        return phoneLink ? phoneLink.textContent : '(310) 438-6866';
    }

    /**
     * Fire conversion tracking events after successful form submission
     */
    function fireConversionEvents(formData) {
        // Google Analytics 4
        if (typeof gtag === 'function') {
            gtag('event', 'generate_lead', {
                event_category: 'Lead Form',
                event_label: formData.get('service') || 'General',
                value: 1
            });
        }

        // Facebook Pixel
        if (typeof fbq === 'function') {
            fbq('track', 'Lead', {
                content_name: formData.get('service') || 'General Inquiry',
                content_category: 'Lead Form'
            });
        }

        // Google Ads Conversion (if configured)
        if (typeof gtag === 'function') {
            // The conversion ID and label are set by the tracking scripts
            var conversionId = document.querySelector('meta[name="ace-gads-conversion"]');
            var conversionLabel = document.querySelector('meta[name="ace-gads-label"]');
            if (conversionId && conversionLabel) {
                gtag('event', 'conversion', {
                    send_to: conversionId.content + '/' + conversionLabel.content
                });
            }
        }
    }

})();
