<?php
/**
 * Lead Form Handler.
 *
 * Handles form submission processing, validation, storage, email
 * notifications, and administrative lead management operations.
 *
 * @package ACE_Theme_Manager
 * @since   1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class ACE_Lead_Form
 *
 * Processes lead capture form submissions via AJAX, stores entries in
 * the database, sends email notifications, and provides static helper
 * methods for querying and managing leads from the admin side.
 */
class ACE_Lead_Form {

	/**
	 * Maximum number of submissions allowed per IP per hour.
	 *
	 * @var int
	 */
	const RATE_LIMIT_MAX = 3;

	/**
	 * Rate limit window in seconds (1 hour).
	 *
	 * @var int
	 */
	const RATE_LIMIT_WINDOW = 3600;

	/**
	 * Valid preferred contact methods.
	 *
	 * @var array
	 */
	const VALID_CONTACT_METHODS = array( 'phone', 'email', 'text' );

	/**
	 * Initialize the class by registering AJAX handlers.
	 *
	 * @return void
	 */
	public function init() {
		add_action( 'wp_ajax_ace_submit_lead', array( $this, 'handle_submission' ) );
		add_action( 'wp_ajax_nopriv_ace_submit_lead', array( $this, 'handle_submission' ) );
	}

	/**
	 * Handle an incoming lead form AJAX submission.
	 *
	 * Performs nonce verification, rate limiting, honeypot detection,
	 * input validation, database insertion, and email notification.
	 *
	 * @return void Outputs JSON response and terminates.
	 */
	public function handle_submission() {
		// 1. Verify nonce.
		check_ajax_referer( 'ace_lead_form_nonce', 'ace_form_nonce' );

		// 2. Rate limiting.
		$ip_hash        = hash( 'sha256', $this->get_client_ip() );
		$transient_key  = 'ace_lead_limit_' . $ip_hash;
		$current_count  = (int) get_transient( $transient_key );

		if ( $current_count >= self::RATE_LIMIT_MAX ) {
			wp_send_json_error(
				array(
					'message' => __( 'Too many submissions. Please try again later.', 'ace-theme-manager' ),
				),
				429
			);
		}

		// 3. Honeypot check - silently return success for bots.
		if ( ! empty( $_POST['website_url'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification
			wp_send_json_success(
				array(
					'message' => __( 'Thank you for your inquiry.', 'ace-theme-manager' ),
				)
			);
		}

		// 4. Sanitize and validate all inputs.
		$errors = array();

		// Full name - required, max 255 chars.
		$full_name = isset( $_POST['full_name'] ) ? sanitize_text_field( wp_unslash( $_POST['full_name'] ) ) : '';
		if ( empty( $full_name ) ) {
			$errors['full_name'] = __( 'Full name is required.', 'ace-theme-manager' );
		} elseif ( mb_strlen( $full_name ) > 255 ) {
			$errors['full_name'] = __( 'Full name must be 255 characters or fewer.', 'ace-theme-manager' );
		}

		// Phone - required, validated format.
		$phone = isset( $_POST['phone'] ) ? sanitize_text_field( wp_unslash( $_POST['phone'] ) ) : '';
		if ( empty( $phone ) ) {
			$errors['phone'] = __( 'Phone number is required.', 'ace-theme-manager' );
		} elseif ( ! preg_match( '/^[\+]?[\d\s\-\(\)\.]{7,20}$/', $phone ) ) {
			$errors['phone'] = __( 'Please enter a valid phone number.', 'ace-theme-manager' );
		}

		// Email - optional, but if provided must be valid.
		$email = isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : '';
		if ( ! empty( $email ) && ! is_email( $email ) ) {
			$errors['email'] = __( 'Please enter a valid email address.', 'ace-theme-manager' );
		}

		// Service - optional.
		$service = isset( $_POST['service'] ) ? sanitize_text_field( wp_unslash( $_POST['service'] ) ) : '';

		// Message - optional, max 2000 chars.
		$message = isset( $_POST['message'] ) ? sanitize_textarea_field( wp_unslash( $_POST['message'] ) ) : '';
		if ( mb_strlen( $message ) > 2000 ) {
			$errors['message'] = __( 'Message must be 2000 characters or fewer.', 'ace-theme-manager' );
		}

		// Preferred contact - must be one of the allowed values.
		$preferred_contact = isset( $_POST['preferred_contact'] ) ? sanitize_text_field( wp_unslash( $_POST['preferred_contact'] ) ) : '';
		if ( ! empty( $preferred_contact ) && ! in_array( $preferred_contact, self::VALID_CONTACT_METHODS, true ) ) {
			$errors['preferred_contact'] = __( 'Invalid contact method selected.', 'ace-theme-manager' );
		}

		// Source page - optional.
		$source_page = isset( $_POST['source_page'] ) ? sanitize_text_field( wp_unslash( $_POST['source_page'] ) ) : '';

		// Source URL - optional.
		$source_url = isset( $_POST['source_url'] ) ? esc_url_raw( wp_unslash( $_POST['source_url'] ) ) : '';

		// 5. Return validation errors if any.
		if ( ! empty( $errors ) ) {
			wp_send_json_error(
				array(
					'message' => __( 'Please correct the errors below.', 'ace-theme-manager' ),
					'errors'  => $errors,
				),
				422
			);
		}

		// 6. Insert into database.
		global $wpdb;

		$table_name = $wpdb->prefix . 'ace_leads';
		$lead_data  = array(
			'full_name'         => $full_name,
			'phone'             => $phone,
			'email'             => $email,
			'service'           => $service,
			'message'           => $message,
			'preferred_contact' => $preferred_contact,
			'source_page'       => $source_page,
			'source_url'        => $source_url,
			'ip_address'        => $this->get_client_ip(),
			'user_agent'        => isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : '',
			'created_at'        => current_time( 'mysql' ),
			'is_read'           => 0,
			'notes'             => '',
		);

		$format = array(
			'%s', // full_name
			'%s', // phone
			'%s', // email
			'%s', // service
			'%s', // message
			'%s', // preferred_contact
			'%s', // source_page
			'%s', // source_url
			'%s', // ip_address
			'%s', // user_agent
			'%s', // created_at
			'%d', // is_read
			'%s', // notes
		);

		$inserted = $wpdb->insert( $table_name, $lead_data, $format );

		if ( false === $inserted ) {
			wp_send_json_error(
				array(
					'message' => __( 'An error occurred while saving your submission. Please try again.', 'ace-theme-manager' ),
				),
				500
			);
		}

		$lead_data['id'] = $wpdb->insert_id;

		// 7. Increment rate limit transient.
		if ( false === get_transient( $transient_key ) ) {
			set_transient( $transient_key, 1, self::RATE_LIMIT_WINDOW );
		} else {
			set_transient( $transient_key, $current_count + 1, self::RATE_LIMIT_WINDOW );
		}

		// 8. Send email notification.
		$this->send_notification( $lead_data );

		// 9. Return success.
		$success_message = get_option( 'ace_form_success_message', __( 'Thank you for your inquiry. We will get back to you shortly.', 'ace-theme-manager' ) );
		$thank_you_url   = get_option( 'ace_form_thank_you_url', '' );

		$response = array(
			'message' => $success_message,
		);

		if ( ! empty( $thank_you_url ) ) {
			$response['redirect'] = esc_url( home_url( $thank_you_url ) );
		}

		wp_send_json_success( $response );
	}

	/**
	 * Send an email notification for a new lead.
	 *
	 * Renders the HTML email template and sends it to the configured
	 * recipient addresses using wp_mail().
	 *
	 * @param array $lead_data Associative array of lead field values.
	 * @return bool Whether the email was sent successfully.
	 */
	public function send_notification( $lead_data ) {
		$recipients = get_option( 'ace_form_recipients', 'info@acedesignbuild.com' );

		// Support multiple comma-separated emails.
		$to = array_map( 'trim', explode( ',', $recipients ) );
		$to = array_filter( $to, 'is_email' );

		if ( empty( $to ) ) {
			return false;
		}

		// Build subject line.
		$service_label = ! empty( $lead_data['service'] ) ? $lead_data['service'] : __( 'General Inquiry', 'ace-theme-manager' );
		$subject       = sprintf(
			/* translators: 1: Service type, 2: Lead full name. */
			__( 'New Lead from ACE Website - %1$s - %2$s', 'ace-theme-manager' ),
			$service_label,
			$lead_data['full_name']
		);

		// Render the email template.
		$template_path = ACE_PLUGIN_PATH . 'templates/email-notification.php';

		if ( ! file_exists( $template_path ) ) {
			return false;
		}

		ob_start();
		include $template_path;
		$body = ob_get_clean();

		// Set headers for HTML email with Reply-To.
		$headers = array(
			'Content-Type: text/html; charset=UTF-8',
		);

		if ( ! empty( $lead_data['email'] ) && is_email( $lead_data['email'] ) ) {
			$reply_to_name = ! empty( $lead_data['full_name'] ) ? $lead_data['full_name'] : '';
			$headers[]     = sprintf( 'Reply-To: %s <%s>', $reply_to_name, $lead_data['email'] );
		}

		return wp_mail( $to, $subject, $body, $headers );
	}

	/**
	 * Query leads from the database with optional filters and pagination.
	 *
	 * @param array $args {
	 *     Optional. Query arguments.
	 *
	 *     @type string $date_from   Start date (Y-m-d format).
	 *     @type string $date_to     End date (Y-m-d format).
	 *     @type string $service     Filter by service type.
	 *     @type int    $is_read     Filter by read status (0 or 1).
	 *     @type int    $per_page    Number of results per page. Default 20.
	 *     @type int    $offset      Number of results to skip. Default 0.
	 *     @type string $orderby     Column to order by. Default 'created_at'.
	 *     @type string $order       Sort direction, ASC or DESC. Default 'DESC'.
	 * }
	 * @return array Array of lead objects.
	 */
	public static function get_leads( $args = array() ) {
		global $wpdb;

		$defaults = array(
			'date_from' => '',
			'date_to'   => '',
			'service'   => '',
			'is_read'   => null,
			'per_page'  => 20,
			'offset'    => 0,
			'orderby'   => 'created_at',
			'order'     => 'DESC',
		);

		$args       = wp_parse_args( $args, $defaults );
		$table_name = $wpdb->prefix . 'ace_leads';

		// Whitelist of allowed columns for ordering.
		$allowed_orderby = array( 'id', 'full_name', 'email', 'service', 'created_at', 'is_read' );
		$orderby         = in_array( $args['orderby'], $allowed_orderby, true ) ? $args['orderby'] : 'created_at';
		$order           = strtoupper( $args['order'] ) === 'ASC' ? 'ASC' : 'DESC';

		$where  = array();
		$values = array();

		// Date range filters.
		if ( ! empty( $args['date_from'] ) ) {
			$where[]  = 'created_at >= %s';
			$values[] = sanitize_text_field( $args['date_from'] ) . ' 00:00:00';
		}

		if ( ! empty( $args['date_to'] ) ) {
			$where[]  = 'created_at <= %s';
			$values[] = sanitize_text_field( $args['date_to'] ) . ' 23:59:59';
		}

		// Service filter.
		if ( ! empty( $args['service'] ) ) {
			$where[]  = 'service = %s';
			$values[] = sanitize_text_field( $args['service'] );
		}

		// Read status filter.
		if ( null !== $args['is_read'] ) {
			$where[]  = 'is_read = %d';
			$values[] = (int) $args['is_read'];
		}

		// Build the WHERE clause.
		$where_sql = '';
		if ( ! empty( $where ) ) {
			$where_sql = 'WHERE ' . implode( ' AND ', $where );
		}

		// Pagination.
		$per_page = absint( $args['per_page'] );
		$offset   = absint( $args['offset'] );

		// Build the full query. Orderby and order are whitelisted above.
		$sql = "SELECT * FROM {$table_name} {$where_sql} ORDER BY {$orderby} {$order} LIMIT %d OFFSET %d";

		$values[] = $per_page;
		$values[] = $offset;

		// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Query is built with whitelisted values.
		$results = $wpdb->get_results( $wpdb->prepare( $sql, $values ) );

		return $results ? $results : array();
	}

	/**
	 * Retrieve a single lead by its ID.
	 *
	 * @param int $id Lead ID.
	 * @return object|false Lead object on success, false on failure.
	 */
	public static function get_lead( $id ) {
		global $wpdb;

		$table_name = $wpdb->prefix . 'ace_leads';
		$id         = absint( $id );

		if ( 0 === $id ) {
			return false;
		}

		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$lead = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$table_name} WHERE id = %d", $id ) );

		return $lead ? $lead : false;
	}

	/**
	 * Mark a lead as read.
	 *
	 * @param int $id Lead ID.
	 * @return bool True on success, false on failure.
	 */
	public static function mark_read( $id ) {
		global $wpdb;

		$table_name = $wpdb->prefix . 'ace_leads';
		$id         = absint( $id );

		if ( 0 === $id ) {
			return false;
		}

		$result = $wpdb->update(
			$table_name,
			array( 'is_read' => 1 ),
			array( 'id' => $id ),
			array( '%d' ),
			array( '%d' )
		);

		return false !== $result;
	}

	/**
	 * Update the notes field for a lead.
	 *
	 * @param int    $id    Lead ID.
	 * @param string $notes New notes content.
	 * @return bool True on success, false on failure.
	 */
	public static function update_notes( $id, $notes ) {
		global $wpdb;

		$table_name = $wpdb->prefix . 'ace_leads';
		$id         = absint( $id );

		if ( 0 === $id ) {
			return false;
		}

		$result = $wpdb->update(
			$table_name,
			array( 'notes' => sanitize_textarea_field( $notes ) ),
			array( 'id' => $id ),
			array( '%s' ),
			array( '%d' )
		);

		return false !== $result;
	}

	/**
	 * Delete a lead by ID.
	 *
	 * Requires the current user to have the manage_options capability.
	 *
	 * @param int $id Lead ID.
	 * @return bool True on success, false on failure or insufficient permissions.
	 */
	public static function delete_lead( $id ) {
		if ( ! current_user_can( 'manage_options' ) ) {
			return false;
		}

		global $wpdb;

		$table_name = $wpdb->prefix . 'ace_leads';
		$id         = absint( $id );

		if ( 0 === $id ) {
			return false;
		}

		$result = $wpdb->delete(
			$table_name,
			array( 'id' => $id ),
			array( '%d' )
		);

		return false !== $result;
	}

	/**
	 * Get the count of leads, optionally filtered by read status.
	 *
	 * @param int|null $is_read Optional. Filter by read status (0 or 1). Null for all.
	 * @return int Number of matching leads.
	 */
	public static function get_lead_count( $is_read = null ) {
		global $wpdb;

		$table_name = $wpdb->prefix . 'ace_leads';

		if ( null !== $is_read ) {
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			$count = $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM {$table_name} WHERE is_read = %d", (int) $is_read ) );
		} else {
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			$count = $wpdb->get_var( "SELECT COUNT(*) FROM {$table_name}" );
		}

		return (int) $count;
	}

	/**
	 * Export all leads as a CSV file download.
	 *
	 * Requires the manage_options capability. Outputs CSV headers and data
	 * directly, then terminates execution.
	 *
	 * @return void Terminates with exit.
	 */
	public function export_csv() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have permission to export leads.', 'ace-theme-manager' ), 403 );
		}

		global $wpdb;

		$table_name = $wpdb->prefix . 'ace_leads';

		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$leads = $wpdb->get_results( "SELECT * FROM {$table_name} ORDER BY created_at DESC", ARRAY_A );

		$filename = 'ace-leads-' . gmdate( 'Y-m-d-His' ) . '.csv';

		// Prevent caching.
		nocache_headers();

		header( 'Content-Type: text/csv; charset=utf-8' );
		header( 'Content-Disposition: attachment; filename=' . $filename );
		header( 'Pragma: no-cache' );
		header( 'Expires: 0' );

		$output = fopen( 'php://output', 'w' );

		if ( false === $output ) {
			wp_die( esc_html__( 'Unable to open output stream for CSV export.', 'ace-theme-manager' ), 500 );
		}

		// CSV column headers.
		fputcsv( $output, array(
			'ID',
			'Name',
			'Phone',
			'Email',
			'Service',
			'Message',
			'Contact Method',
			'Source Page',
			'IP',
			'Date',
			'Read',
			'Notes',
		) );

		// Data rows.
		if ( $leads ) {
			foreach ( $leads as $lead ) {
				fputcsv( $output, array(
					$lead['id'],
					$lead['full_name'],
					$lead['phone'],
					$lead['email'],
					$lead['service'],
					$lead['message'],
					$lead['preferred_contact'],
					$lead['source_page'],
					$lead['ip_address'],
					$lead['created_at'],
					$lead['is_read'] ? 'Yes' : 'No',
					$lead['notes'],
				) );
			}
		}

		fclose( $output );
		exit;
	}

	/**
	 * Get the client IP address.
	 *
	 * Checks common proxy headers before falling back to REMOTE_ADDR.
	 * The result is always validated as a proper IP address.
	 *
	 * @return string Client IP address or empty string if unavailable.
	 */
	private function get_client_ip() {
		$ip = '';

		$headers = array(
			'HTTP_CF_CONNECTING_IP', // Cloudflare.
			'HTTP_X_FORWARDED_FOR',
			'HTTP_X_REAL_IP',
			'REMOTE_ADDR',
		);

		foreach ( $headers as $header ) {
			if ( ! empty( $_SERVER[ $header ] ) ) {
				// X-Forwarded-For may contain a chain - use the first IP.
				$ip_list = explode( ',', sanitize_text_field( wp_unslash( $_SERVER[ $header ] ) ) );
				$ip      = trim( $ip_list[0] );
				break;
			}
		}

		// Validate the IP address.
		if ( false === filter_var( $ip, FILTER_VALIDATE_IP ) ) {
			$ip = '';
		}

		return $ip;
	}
}
