/**
 * ACE Design & Build - Main JavaScript
 * Pure vanilla JS - no dependencies
 * @version 1.0.0
 */

(function() {
    'use strict';

    // Wait for DOM ready
    document.addEventListener('DOMContentLoaded', function() {
        initHeader();
        initMobileMenu();
        initScrollReveal();
        initStatsCounter();
        initTestimonialSlider();
        initFaqAccordion();
        initPortfolioFilter();
        initAnnouncementBar();
        initMobileSubmenu();
        initBackToTop();
        initConsultationPopup();
        initSmoothScrollLinks();
    });

    /**
     * Sticky Header - shrink on scroll, add shadow
     */
    function initHeader() {
        var header = document.getElementById('site-header');
        if (!header) return;

        var scrollThreshold = 100;
        var lastScroll = 0;

        function handleScroll() {
            var currentScroll = window.pageYOffset || document.documentElement.scrollTop;

            if (currentScroll > scrollThreshold) {
                header.classList.add('scrolled');
            } else {
                header.classList.remove('scrolled');
            }

            lastScroll = currentScroll;
        }

        // Throttle scroll handler
        var ticking = false;
        window.addEventListener('scroll', function() {
            if (!ticking) {
                window.requestAnimationFrame(function() {
                    handleScroll();
                    ticking = false;
                });
                ticking = true;
            }
        }, { passive: true });

        // Initial check
        handleScroll();
    }

    /**
     * Mobile Menu - hamburger toggle, overlay
     */
    function initMobileMenu() {
        var toggle = document.querySelector('.mobile-menu-toggle');
        var overlay = document.querySelector('.mobile-nav-overlay');
        var close = document.querySelector('.mobile-nav-close');

        if (!toggle || !overlay) return;

        function openMenu() {
            overlay.classList.add('active');
            overlay.setAttribute('aria-hidden', 'false');
            toggle.setAttribute('aria-expanded', 'true');
            document.body.style.overflow = 'hidden';
        }

        function closeMenu() {
            overlay.classList.remove('active');
            overlay.setAttribute('aria-hidden', 'true');
            toggle.setAttribute('aria-expanded', 'false');
            document.body.style.overflow = '';
        }

        toggle.addEventListener('click', openMenu);
        if (close) close.addEventListener('click', closeMenu);

        // Close on overlay click (outside content)
        overlay.addEventListener('click', function(e) {
            if (e.target === overlay) closeMenu();
        });

        // Close on Escape key
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape' && overlay.classList.contains('active')) {
                closeMenu();
            }
        });

        // Close menu on link click
        var menuLinks = overlay.querySelectorAll('a:not(.has-submenu > a)');
        menuLinks.forEach(function(link) {
            link.addEventListener('click', closeMenu);
        });
    }

    /**
     * Mobile Submenu Toggle
     */
    function initMobileSubmenu() {
        var subMenuToggles = document.querySelectorAll('.mobile-nav-menu .has-submenu > a');

        subMenuToggles.forEach(function(toggle) {
            toggle.addEventListener('click', function(e) {
                e.preventDefault();
                var submenu = this.nextElementSibling;
                if (submenu && submenu.classList.contains('mobile-submenu')) {
                    submenu.classList.toggle('open');
                    // Rotate chevron
                    var icon = this.querySelector('.ace-icon');
                    if (icon) {
                        icon.style.transform = submenu.classList.contains('open') ? 'rotate(180deg)' : '';
                    }
                }
            });
        });
    }

    /**
     * Scroll Reveal - fade-in-up animation on scroll using Intersection Observer
     */
    function initScrollReveal() {
        var sections = document.querySelectorAll('.section');

        if (!('IntersectionObserver' in window)) {
            // Fallback: show everything
            sections.forEach(function(el) { el.classList.add('revealed'); });
            return;
        }

        var observer = new IntersectionObserver(function(entries) {
            entries.forEach(function(entry) {
                if (entry.isIntersecting) {
                    entry.target.classList.add('scroll-reveal', 'revealed');
                    observer.unobserve(entry.target);
                }
            });
        }, {
            threshold: 0.1,
            rootMargin: '0px 0px -50px 0px'
        });

        sections.forEach(function(section) {
            section.classList.add('scroll-reveal');
            observer.observe(section);
        });
    }

    /**
     * Stats Counter - animate counting up when visible
     */
    function initStatsCounter() {
        var statNumbers = document.querySelectorAll('.stat-number[data-target]');
        if (statNumbers.length === 0) return;

        if (!('IntersectionObserver' in window)) {
            statNumbers.forEach(function(el) {
                el.textContent = el.dataset.target + (el.dataset.suffix || '');
            });
            return;
        }

        var animated = false;

        var observer = new IntersectionObserver(function(entries) {
            entries.forEach(function(entry) {
                if (entry.isIntersecting && !animated) {
                    animated = true;
                    animateCounters();
                    observer.disconnect();
                }
            });
        }, { threshold: 0.3 });

        // Observe the stats section
        var statsSection = document.querySelector('.stats-section');
        if (statsSection) {
            observer.observe(statsSection);
        }

        function animateCounters() {
            statNumbers.forEach(function(el) {
                var target = parseFloat(el.dataset.target);
                var suffix = el.dataset.suffix || '';
                var duration = 2000;
                var startTime = null;
                var isDecimal = target % 1 !== 0;

                function animate(currentTime) {
                    if (!startTime) startTime = currentTime;
                    var progress = Math.min((currentTime - startTime) / duration, 1);

                    // Ease out cubic
                    var eased = 1 - Math.pow(1 - progress, 3);
                    var current = eased * target;

                    if (isDecimal) {
                        el.textContent = current.toFixed(1) + suffix;
                    } else {
                        el.textContent = Math.floor(current).toLocaleString() + suffix;
                    }

                    if (progress < 1) {
                        requestAnimationFrame(animate);
                    } else {
                        if (isDecimal) {
                            el.textContent = target.toFixed(1) + suffix;
                        } else {
                            el.textContent = target.toLocaleString() + suffix;
                        }
                    }
                }

                requestAnimationFrame(animate);
            });
        }
    }

    /**
     * Testimonial Slider
     */
    function initTestimonialSlider() {
        var cards = document.querySelectorAll('.testimonials-slider .testimonial-card');
        var dots = document.querySelectorAll('.testimonials-dots .dot');
        var prevBtn = document.querySelector('.testimonial-prev');
        var nextBtn = document.querySelector('.testimonial-next');

        if (cards.length === 0) return;

        var currentSlide = 0;
        var totalSlides = cards.length;
        var autoPlayInterval;

        function showSlide(index) {
            if (index < 0) index = totalSlides - 1;
            if (index >= totalSlides) index = 0;

            currentSlide = index;

            cards.forEach(function(card, i) {
                card.classList.toggle('active', i === currentSlide);
            });

            dots.forEach(function(dot, i) {
                dot.classList.toggle('active', i === currentSlide);
            });
        }

        function nextSlide() {
            showSlide(currentSlide + 1);
        }

        function prevSlide() {
            showSlide(currentSlide - 1);
        }

        if (nextBtn) nextBtn.addEventListener('click', function() {
            nextSlide();
            resetAutoPlay();
        });

        if (prevBtn) prevBtn.addEventListener('click', function() {
            prevSlide();
            resetAutoPlay();
        });

        dots.forEach(function(dot) {
            dot.addEventListener('click', function() {
                var slideIndex = parseInt(this.dataset.slide, 10);
                showSlide(slideIndex);
                resetAutoPlay();
            });
        });

        // Auto-play
        function startAutoPlay() {
            autoPlayInterval = setInterval(nextSlide, 5000);
        }

        function resetAutoPlay() {
            clearInterval(autoPlayInterval);
            startAutoPlay();
        }

        // Touch/swipe support
        var startX = 0;
        var slider = document.querySelector('.testimonials-slider');
        if (slider) {
            slider.addEventListener('touchstart', function(e) {
                startX = e.touches[0].clientX;
            }, { passive: true });

            slider.addEventListener('touchend', function(e) {
                var endX = e.changedTouches[0].clientX;
                var diff = startX - endX;
                if (Math.abs(diff) > 50) {
                    if (diff > 0) nextSlide();
                    else prevSlide();
                    resetAutoPlay();
                }
            }, { passive: true });
        }

        startAutoPlay();
    }

    /**
     * FAQ Accordion
     */
    function initFaqAccordion() {
        var questions = document.querySelectorAll('.faq-question');

        questions.forEach(function(question) {
            question.addEventListener('click', function() {
                var answer = this.nextElementSibling;
                var isOpen = this.getAttribute('aria-expanded') === 'true';

                // Close all others
                questions.forEach(function(q) {
                    q.setAttribute('aria-expanded', 'false');
                    var a = q.nextElementSibling;
                    if (a) a.style.maxHeight = null;
                });

                // Toggle current
                if (!isOpen) {
                    this.setAttribute('aria-expanded', 'true');
                    if (answer) {
                        answer.style.maxHeight = answer.scrollHeight + 'px';
                    }
                }
            });
        });
    }

    /**
     * Portfolio Filter
     */
    function initPortfolioFilter() {
        var filterBtns = document.querySelectorAll('.filter-btn');
        var items = document.querySelectorAll('.portfolio-item');

        if (filterBtns.length === 0 || items.length === 0) return;

        filterBtns.forEach(function(btn) {
            btn.addEventListener('click', function() {
                var filter = this.dataset.filter;

                // Update active button
                filterBtns.forEach(function(b) { b.classList.remove('active'); });
                this.classList.add('active');

                // Filter items
                items.forEach(function(item) {
                    var categories = (item.dataset.categories || '').split(',');
                    if (filter === 'all' || categories.indexOf(filter) !== -1) {
                        item.classList.remove('hidden');
                        item.style.display = '';
                    } else {
                        item.classList.add('hidden');
                        item.style.display = 'none';
                    }
                });
            });
        });
    }

    /**
     * Back to Top Button - show after 400px scroll, smooth scroll to top
     */
    function initBackToTop() {
        var btn = document.getElementById('back-to-top');
        if (!btn) return;

        var scrollThreshold = 400;

        // Show/hide button based on scroll position
        var ticking = false;
        window.addEventListener('scroll', function() {
            if (!ticking) {
                window.requestAnimationFrame(function() {
                    var currentScroll = window.pageYOffset || document.documentElement.scrollTop;
                    if (currentScroll > scrollThreshold) {
                        btn.classList.add('visible');
                    } else {
                        btn.classList.remove('visible');
                    }
                    ticking = false;
                });
                ticking = true;
            }
        }, { passive: true });

        // Smooth scroll to top on click
        btn.addEventListener('click', function() {
            window.scrollTo({
                top: 0,
                behavior: 'smooth'
            });
        });
    }

    /**
     * Consultation Popup - triggered by all CTA buttons with data-popup="consultation"
     */
    function initConsultationPopup() {
        var popup = document.getElementById('consultationPopup');
        if (!popup) return;

        var overlay = popup.querySelector('.popup-overlay');
        var closeBtn = popup.querySelector('.popup-close');
        var popupContent = popup.querySelector('.popup-content');

        function openPopup() {
            popup.classList.add('open');
            document.body.style.overflow = 'hidden';
            // Focus the first input
            setTimeout(function() {
                var firstInput = popup.querySelector('input[name="full_name"]');
                if (firstInput) firstInput.focus();
            }, 400);
        }

        function closePopup() {
            popup.classList.remove('open');
            document.body.style.overflow = '';
        }

        // All CTA buttons trigger popup
        document.querySelectorAll('[data-popup="consultation"]').forEach(function(btn) {
            btn.addEventListener('click', function(e) {
                e.preventDefault();
                // Close mobile menu if open
                var mobileOverlay = document.querySelector('.mobile-nav-overlay.active');
                if (mobileOverlay) {
                    mobileOverlay.classList.remove('active');
                    mobileOverlay.setAttribute('aria-hidden', 'true');
                    var toggle = document.querySelector('.mobile-menu-toggle');
                    if (toggle) toggle.setAttribute('aria-expanded', 'false');
                }
                openPopup();
            });
        });

        // Close handlers
        if (closeBtn) closeBtn.addEventListener('click', closePopup);
        if (overlay) overlay.addEventListener('click', closePopup);

        // ESC key
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape' && popup.classList.contains('open')) {
                closePopup();
            }
        });

        // Prevent popup content click from closing
        if (popupContent) {
            popupContent.addEventListener('click', function(e) {
                e.stopPropagation();
            });
        }

        // Handle popup form submission success - auto-close after 3s
        var popupForm = popup.querySelector('.ace-lead-form');
        if (popupForm) {
            var originalOnload = null;
            popupForm.addEventListener('submit', function() {
                // Watch for success message
                var checkSuccess = setInterval(function() {
                    var msg = popupForm.querySelector('.form-message.success');
                    if (msg && msg.style.display !== 'none') {
                        clearInterval(checkSuccess);
                        setTimeout(function() {
                            closePopup();
                            // Reset form and hide message after closing
                            setTimeout(function() {
                                popupForm.reset();
                                msg.style.display = 'none';
                            }, 500);
                        }, 3000);
                    }
                }, 200);
                // Clear interval after 15s to prevent infinite loop
                setTimeout(function() { clearInterval(checkSuccess); }, 15000);
            });
        }
    }

    /**
     * Announcement Bar Close
     */
    function initAnnouncementBar() {
        var closeBtn = document.querySelector('.announcement-close');
        if (!closeBtn) return;

        closeBtn.addEventListener('click', function() {
            var bar = this.closest('.announcement-bar');
            if (bar) {
                bar.style.display = 'none';
                // Store in session
                try {
                    sessionStorage.setItem('ace_announcement_closed', '1');
                } catch(e) {}
            }
        });

        // Check if previously closed in this session
        try {
            if (sessionStorage.getItem('ace_announcement_closed') === '1') {
                var bar = document.querySelector('.announcement-bar');
                if (bar) bar.style.display = 'none';
            }
        } catch(e) {}
    }

    /**
     * Smooth Scroll for data-scroll links
     * Links with data-scroll="true" and href starting with # will smooth-scroll
     * to the target element instead of navigating.
     */
    function initSmoothScrollLinks() {
        document.addEventListener('click', function(e) {
            var link = e.target.closest('[data-scroll="true"]');
            if (!link) return;

            var href = link.getAttribute('href');
            if (!href || href.charAt(0) !== '#') return;

            e.preventDefault();
            var targetId = href.substring(1);
            var target = document.getElementById(targetId);
            if (target) {
                target.scrollIntoView({ behavior: 'smooth' });
            }

            // Close mobile menu if open
            var overlay = document.querySelector('.mobile-nav-overlay.active');
            if (overlay) {
                overlay.classList.remove('active');
                overlay.setAttribute('aria-hidden', 'true');
                document.body.style.overflow = '';
                var toggle = document.querySelector('.mobile-menu-toggle');
                if (toggle) toggle.setAttribute('aria-expanded', 'false');
            }
        });
    }

})();

/* ---- Service Page Testimonial Carousel ---- */
(function() {
    document.addEventListener('DOMContentLoaded', function() {
        document.querySelectorAll('.testimonial-carousel').forEach(function(carousel) {
            var cards = carousel.querySelectorAll('.service-testimonial-card');
            var dots = carousel.querySelectorAll('.carousel-dot');
            var prevBtn = carousel.querySelector('.carousel-prev');
            var nextBtn = carousel.querySelector('.carousel-next');
            var current = 0;
            var total = cards.length;
            var isAnimating = false;

            function goTo(index, direction) {
                if (isAnimating || index === current || total <= 1) return;
                isAnimating = true;
                var exitClass = direction === 'next' ? 'exit-left' : 'exit-right';

                cards[current].classList.add(exitClass);
                cards[current].classList.remove('active');
                if (dots.length) {
                    dots[current].classList.remove('active');
                    dots[index].classList.add('active');
                }

                setTimeout(function() {
                    cards[current].classList.remove(exitClass);
                    current = index;
                    cards[current].style.transform = direction === 'next' ? 'translateX(40px)' : 'translateX(-40px)';
                    cards[current].classList.add('active');
                    requestAnimationFrame(function() {
                        requestAnimationFrame(function() {
                            cards[current].style.transform = '';
                            isAnimating = false;
                        });
                    });
                }, 250);
            }

            if (nextBtn) nextBtn.addEventListener('click', function() {
                goTo((current + 1) % total, 'next');
            });
            if (prevBtn) prevBtn.addEventListener('click', function() {
                goTo((current - 1 + total) % total, 'prev');
            });
            dots.forEach(function(dot) {
                dot.addEventListener('click', function() {
                    var idx = parseInt(this.getAttribute('data-index'));
                    goTo(idx, idx > current ? 'next' : 'prev');
                });
            });

            // Auto-advance every 6 seconds, pause on hover.
            var auto = setInterval(function() { goTo((current + 1) % total, 'next'); }, 6000);
            carousel.addEventListener('mouseenter', function() { clearInterval(auto); });
            carousel.addEventListener('mouseleave', function() {
                auto = setInterval(function() { goTo((current + 1) % total, 'next'); }, 6000);
            });
        });
    });
})();
