<?php
/**
 * Service page URL routing
 *
 * Handles /services/{slug}/ URLs and routes them to the service page template.
 *
 * Note: This file must be included from functions.php:
 *   require_once get_template_directory() . '/inc/service-routing.php';
 *
 * @package ACE_Theme
 */

// Add rewrite rules for service pages
function ace_service_rewrite_rules() {
    add_rewrite_rule(
        'services/([^/]+)/?$',
        'index.php?ace_service=$matches[1]',
        'top'
    );
}
add_action( 'init', 'ace_service_rewrite_rules' );

// Register query var
function ace_service_query_vars( $vars ) {
    $vars[] = 'ace_service';
    return $vars;
}
add_filter( 'query_vars', 'ace_service_query_vars' );

// Load service template when ace_service query var is set
function ace_service_template( $template ) {
    $service_slug = get_query_var( 'ace_service' );
    if ( $service_slug ) {
        $service = ace_get_service( $service_slug );
        if ( $service ) {
            $service_template = locate_template( 'template-parts/service-page.php' );
            if ( $service_template ) {
                return $service_template;
            }
        }
    }
    return $template;
}
add_filter( 'template_include', 'ace_service_template' );

// Set proper page title for service pages
function ace_service_document_title( $title ) {
    $service_slug = get_query_var( 'ace_service' );
    if ( $service_slug ) {
        $service = ace_get_service( $service_slug );
        if ( $service ) {
            $seo = get_option( 'ace_seo', array() );
            $seo_key = str_replace( '-', '_', $service_slug );
            if ( isset( $seo[ $seo_key . '_title' ] ) && $seo[ $seo_key . '_title' ] ) {
                $title['title'] = $seo[ $seo_key . '_title' ];
            } else {
                $title['title'] = $service['h1'] . ' | ' . ace_get_company_name();
            }
            // Remove tagline/site parts
            unset( $title['tagline'] );
            unset( $title['site'] );
        }
    }
    return $title;
}
add_filter( 'document_title_parts', 'ace_service_document_title' );

// SEO meta for service pages
function ace_service_meta() {
    $service_slug = get_query_var( 'ace_service' );
    if ( ! $service_slug ) return;

    $service = ace_get_service( $service_slug );
    if ( ! $service ) return;

    $seo = get_option( 'ace_seo', array() );
    $seo_key = str_replace( '-', '_', $service_slug );

    $description = '';
    if ( isset( $seo[ $seo_key . '_description' ] ) ) {
        $description = $seo[ $seo_key . '_description' ];
    } elseif ( isset( $service['meta_description'] ) ) {
        $description = $service['meta_description'];
    }

    if ( $description ) {
        echo '<meta name="description" content="' . esc_attr( $description ) . '">' . "\n";
    }
}
add_action( 'wp_head', 'ace_service_meta', 2 );

// Flush rewrite rules when needed
function ace_flush_service_rules() {
    ace_service_rewrite_rules();
    flush_rewrite_rules();
}
register_activation_hook( __FILE__, 'ace_flush_service_rules' );
