<?php
/**
 * Template helper functions for ACE Theme
 */

/**
 * Get theme option with fallback
 * Works with the ACE Theme Manager plugin options
 */
function ace_get_option( $key, $default = '' ) {
    $value = get_option( 'ace_' . $key, $default );
    return ( $value !== '' && $value !== false ) ? $value : $default;
}

/**
 * Get default icon mappings for the entire site.
 */
function ace_get_default_icons() {
    return array(
        'services' => array(
            'kitchen-remodeling'        => 'fa-utensils',
            'bathroom-remodeling'       => 'fa-shower',
            'full-home-remodeling'      => 'fa-hammer',
            'adu'                       => 'fa-building',
            'home-additions'            => 'fa-maximize',
            'backyard-remodeling'       => 'fa-tree',
            'landscape-and-hardscaping' => 'fa-seedling',
            'concrete'                  => 'fa-cube',
            'swimming-pools'            => 'fa-water-ladder',
            'outdoor-kitchens'          => 'fa-fire-burner',
            'patio-covers'              => 'fa-umbrella',
            'pavers'                    => 'fa-border-all',
            'driveways'                 => 'fa-road',
            'artificial-turf'           => 'fa-spa',
            'front-yard-renovation'     => 'fa-house',
        ),
        'values' => array(
            'craftsmanship'  => 'fa-gem',
            'transparency'   => 'fa-eye',
            'respect'        => 'fa-heart',
            'accountability' => 'fa-medal',
        ),
        'trust' => array(
            'rating'     => 'fa-star',
            'licensed'   => 'fa-shield-halved',
            'experience' => 'fa-trophy',
            'area'       => 'fa-location-dot',
        ),
        'contact' => array(
            'phone'   => 'fa-phone',
            'email'   => 'fa-envelope',
            'address' => 'fa-location-dot',
            'hours'   => 'fa-clock',
        ),
        'why_ace' => array(
            'design_build'  => 'fa-compass-drafting',
            'pricing'       => 'fa-hand-holding-dollar',
            'licensed'      => 'fa-shield-halved',
            'experience'    => 'fa-trophy',
        ),
        'lead_trust' => array(
            'licensed'    => 'fa-shield-halved',
            'estimates'   => 'fa-hand-holding-dollar',
            'years'       => 'fa-trophy',
            'guaranteed'  => 'fa-circle-check',
        ),
    );
}

/**
 * Get icon HTML for a group/key combination.
 *
 * @param string $group  Icon group (services, values, trust, contact, why_ace, lead_trust).
 * @param string $key    Icon key within the group.
 * @param string $style  Rendering style: container, large, inline, nav, raw.
 * @return string HTML markup.
 */
function ace_get_icon( $group, $key, $style = 'container' ) {
    $icons    = get_option( 'ace_icons', array() );
    $defaults = ace_get_default_icons();

    // Look up: saved option first, then defaults, then fallback.
    $icon_class = 'fa-circle';
    if ( isset( $icons[ $group ][ $key ] ) && $icons[ $group ][ $key ] ) {
        $icon_class = $icons[ $group ][ $key ];
    } elseif ( isset( $defaults[ $group ][ $key ] ) ) {
        $icon_class = $defaults[ $group ][ $key ];
    }

    $fa = '<i class="fas ' . esc_attr( $icon_class ) . '"></i>';

    switch ( $style ) {
        case 'container':
            return '<div class="ace-icon-container">' . $fa . '</div>';
        case 'large':
            return '<div class="ace-icon-container-lg">' . $fa . '</div>';
        case 'inline':
            return '<span class="ace-icon-inline">' . $fa . '</span>';
        case 'nav':
            return '<span class="ace-nav-icon">' . $fa . '</span>';
        case 'raw':
        default:
            return $fa;
    }
}

/**
 * Get the FA icon class for a service by slug.
 */
function ace_get_service_icon_class( $slug ) {
    $icons    = get_option( 'ace_icons', array() );
    $defaults = ace_get_default_icons();

    if ( isset( $icons['services'][ $slug ] ) && $icons['services'][ $slug ] ) {
        return $icons['services'][ $slug ];
    }
    if ( isset( $defaults['services'][ $slug ] ) ) {
        return $defaults['services'][ $slug ];
    }
    return 'fa-house';
}

/**
 * Get unique FA icons for each "What's Included" feature card per service.
 */
function ace_get_feature_card_icons( $slug ) {
    $map = array(
        'kitchen-remodeling' => array(
            'fa-ruler-combined', 'fa-cube', 'fa-expand', 'fa-dumpster',
            'fa-bolt', 'fa-faucet', 'fa-box-open', 'fa-table',
            'fa-border-top-left', 'fa-blender', 'fa-square', 'fa-paint-roller',
        ),
        'bathroom-remodeling' => array(
            'fa-drafting-compass', 'fa-droplet', 'fa-faucet-drip', 'fa-border-all',
            'fa-warehouse', 'fa-lightbulb', 'fa-fan', 'fa-glass-water',
            'fa-temperature-low', 'fa-paint-roller',
        ),
        'full-home-remodeling' => array(
            'fa-pencil-ruler', 'fa-compass-drafting', 'fa-helmet-safety', 'fa-file-contract',
            'fa-dumpster', 'fa-bolt', 'fa-faucet', 'fa-hammer',
            'fa-box-open', 'fa-square', 'fa-paint-roller', 'fa-key',
        ),
        'adu' => array(
            'fa-file-contract', 'fa-compass-drafting', 'fa-house-chimney', 'fa-plug',
            'fa-fire-flame-curved', 'fa-door-open', 'fa-utensils', 'fa-square-parking',
            'fa-chart-line',
        ),
        'home-additions' => array(
            'fa-compass-drafting', 'fa-file-contract', 'fa-helmet-safety', 'fa-expand',
            'fa-hammer', 'fa-bolt', 'fa-faucet', 'fa-paint-roller',
            'fa-door-open', 'fa-key',
        ),
        'backyard-remodeling' => array(
            'fa-compass-drafting', 'fa-dumpster', 'fa-cubes', 'fa-seedling',
            'fa-lightbulb', 'fa-fire-burner', 'fa-umbrella', 'fa-water-ladder',
            'fa-couch', 'fa-fence', 'fa-paint-roller',
        ),
        'landscape-and-hardscaping' => array(
            'fa-compass-drafting', 'fa-mountain', 'fa-seedling', 'fa-droplet',
            'fa-lightbulb', 'fa-cubes', 'fa-road', 'fa-fire',
            'fa-fence', 'fa-tree',
        ),
        'concrete' => array(
            'fa-magnifying-glass', 'fa-mountain', 'fa-bars', 'fa-truck-moving',
            'fa-ruler', 'fa-brush', 'fa-road', 'fa-water',
        ),
        'swimming-pools' => array(
            'fa-compass-drafting', 'fa-file-contract', 'fa-truck-moving', 'fa-water',
            'fa-pump-soap', 'fa-temperature-half', 'fa-lightbulb', 'fa-umbrella',
            'fa-shield', 'fa-wrench',
        ),
        'outdoor-kitchens' => array(
            'fa-compass-drafting', 'fa-fire-burner', 'fa-sink', 'fa-plug',
            'fa-cubes', 'fa-umbrella', 'fa-lightbulb', 'fa-blender',
            'fa-couch', 'fa-paint-roller',
        ),
        'patio-covers' => array(
            'fa-compass-drafting', 'fa-helmet-safety', 'fa-cubes', 'fa-umbrella',
            'fa-lightbulb', 'fa-fan', 'fa-paint-roller', 'fa-wrench',
        ),
        'pavers' => array(
            'fa-compass-drafting', 'fa-mountain', 'fa-border-all', 'fa-ruler',
            'fa-truck-moving', 'fa-brush', 'fa-road', 'fa-droplet',
        ),
        'driveways' => array(
            'fa-magnifying-glass', 'fa-mountain', 'fa-truck-moving', 'fa-road',
            'fa-ruler', 'fa-brush', 'fa-droplet', 'fa-wrench',
        ),
        'artificial-turf' => array(
            'fa-magnifying-glass', 'fa-dumpster', 'fa-mountain', 'fa-spa',
            'fa-droplet', 'fa-border-all', 'fa-child', 'fa-dog',
        ),
        'front-yard-renovation' => array(
            'fa-compass-drafting', 'fa-dumpster', 'fa-road', 'fa-seedling',
            'fa-lightbulb', 'fa-cubes', 'fa-droplet', 'fa-paint-roller',
        ),
    );

    if ( isset( $map[ $slug ] ) ) {
        return $map[ $slug ];
    }

    // Generic fallback - a set of diverse icons.
    return array(
        'fa-compass-drafting', 'fa-ruler', 'fa-helmet-safety', 'fa-hammer',
        'fa-bolt', 'fa-faucet', 'fa-paint-roller', 'fa-key',
        'fa-wrench', 'fa-brush', 'fa-cube', 'fa-shield',
    );
}

/**
 * Get company phone number
 */
function ace_get_phone() {
    return ace_get_option( 'phone', '(310) 438-6866' );
}

/**
 * Get phone link (tel: format)
 */
function ace_get_phone_link() {
    $phone = ace_get_phone();
    return 'tel:' . preg_replace( '/[^0-9+]/', '', $phone );
}

/**
 * Get company email
 */
function ace_get_email() {
    return ace_get_option( 'email', 'info@acedesignbuild.com' );
}

/**
 * Get company address
 */
function ace_get_address() {
    return ace_get_option( 'address', '6100 Wilshire Blvd #1200, Los Angeles, CA 90048, USA' );
}

/**
 * Get business hours
 */
function ace_get_hours() {
    return ace_get_option( 'business_hours', 'Monday - Sunday, 10:00 AM - 10:00 PM' );
}

/**
 * Get company name
 */
function ace_get_company_name() {
    return ace_get_option( 'company_name', 'ACE Design & Build' );
}

/**
 * Get tagline
 */
function ace_get_tagline() {
    return ace_get_option( 'tagline', 'Create with Confidence' );
}

/**
 * Get logo URL from plugin option or custom logo
 */
function ace_get_logo_url() {
    $logo = ace_get_option( 'logo', '' );
    if ( $logo ) {
        return $logo;
    }
    $custom_logo_id = get_theme_mod( 'custom_logo' );
    if ( $custom_logo_id ) {
        return wp_get_attachment_image_url( $custom_logo_id, 'full' );
    }
    return ACE_THEME_URI . '/assets/images/logo-placeholder.svg';
}

/**
 * Get white logo URL
 */
function ace_get_logo_white_url() {
    return ace_get_option( 'logo_white', ace_get_logo_url() );
}

/**
 * Output favicon from plugin settings
 */
function ace_output_favicon() {
    $favicon = ace_get_option( 'favicon', '' );
    if ( $favicon ) {
        echo '<link rel="icon" href="' . esc_url( $favicon ) . '" type="image/x-icon">' . "\n";
        echo '<link rel="shortcut icon" href="' . esc_url( $favicon ) . '" type="image/x-icon">' . "\n";
    }
}
add_action( 'wp_head', 'ace_output_favicon', 5 );

/**
 * Get social media URLs
 */
function ace_get_social_links() {
    return array(
        'facebook'  => ace_get_option( 'social_facebook', '' ),
        'instagram' => ace_get_option( 'social_instagram', '' ),
        'houzz'     => ace_get_option( 'social_houzz', '' ),
        'google'    => ace_get_option( 'social_google', '' ),
        'yelp'      => ace_get_option( 'social_yelp', '' ),
        'youtube'   => ace_get_option( 'social_youtube', '' ),
    );
}

/**
 * Render SVG icon by name
 * Icons: phone, email, clock, map-pin, check-circle, star, shield, award, home, hammer, ruler, users, arrow-right, quote, play-circle, calendar, dollar-sign, menu, x, chevron-down, facebook, instagram, youtube, yelp
 */
function ace_icon( $name, $size = 24, $class = '' ) {
    $icons = array(
        'phone' => '<path d="M22 16.92v3a2 2 0 0 1-2.18 2 19.79 19.79 0 0 1-8.63-3.07 19.5 19.5 0 0 1-6-6 19.79 19.79 0 0 1-3.07-8.67A2 2 0 0 1 4.11 2h3a2 2 0 0 1 2 1.72 12.84 12.84 0 0 0 .7 2.81 2 2 0 0 1-.45 2.11L8.09 9.91a16 16 0 0 0 6 6l1.27-1.27a2 2 0 0 1 2.11-.45 12.84 12.84 0 0 0 2.81.7A2 2 0 0 1 22 16.92z"/>',
        'email' => '<path d="M4 4h16c1.1 0 2 .9 2 2v12c0 1.1-.9 2-2 2H4c-1.1 0-2-.9-2-2V6c0-1.1.9-2 2-2z"/><polyline points="22,6 12,13 2,6"/>',
        'clock' => '<circle cx="12" cy="12" r="10"/><polyline points="12 6 12 12 16 14"/>',
        'map-pin' => '<path d="M21 10c0 7-9 13-9 13s-9-6-9-13a9 9 0 0 1 18 0z"/><circle cx="12" cy="10" r="3"/>',
        'check-circle' => '<path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"/><polyline points="22 4 12 14.01 9 11.01"/>',
        'star' => '<polygon points="12 2 15.09 8.26 22 9.27 17 14.14 18.18 21.02 12 17.77 5.82 21.02 7 14.14 2 9.27 8.91 8.26 12 2"/>',
        'shield' => '<path d="M12 22s8-4 8-10V5l-8-3-8 3v7c0 6 8 10 8 10z"/>',
        'award' => '<circle cx="12" cy="8" r="7"/><polyline points="8.21 13.89 7 23 12 20 17 23 15.79 13.88"/>',
        'home' => '<path d="M3 9l9-7 9 7v11a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2z"/><polyline points="9 22 9 12 15 12 15 22"/>',
        'hammer' => '<path d="M15 12l-8.5 8.5c-.83.83-2.17.83-3 0 0 0 0 0 0 0a2.12 2.12 0 0 1 0-3L12 9"/><path d="M17.64 15L22 10.64"/><path d="M20.91 11.7l-1.25-1.25c-.6-.6-.93-1.4-.93-2.25V6.5l-3-3-2.3 2.3 1.6 1.6c.6.6.6 1.57 0 2.17L14 10.6"/>',
        'ruler' => '<path d="M21.21 15.89A1 1 0 0 0 22 15V9a1 1 0 0 0-.29-.71l-4-4a1 1 0 0 0-1.42 0l-8 8a1 1 0 0 0 0 1.42l4 4A1 1 0 0 0 13 18h6a1 1 0 0 0 .71-.29zM6.5 9L9 11.5"/><line x1="10" y1="5" x2="12" y2="7"/><line x1="3" y1="12" x2="5" y2="14"/>',
        'users' => '<path d="M17 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"/><circle cx="9" cy="7" r="4"/><path d="M23 21v-2a4 4 0 0 0-3-3.87"/><path d="M16 3.13a4 4 0 0 1 0 7.75"/>',
        'arrow-right' => '<line x1="5" y1="12" x2="19" y2="12"/><polyline points="12 5 19 12 12 19"/>',
        'quote' => '<path d="M3 21c3 0 7-1 7-8V5c0-1.25-.756-2.017-2-2H4c-1.25 0-2 .75-2 1.972V11c0 1.25.75 2 2 2 1 0 1 0 1 1v1c0 1-1 2-2 2s-1 .008-1 1.031V21z"/><path d="M15 21c3 0 7-1 7-8V5c0-1.25-.757-2.017-2-2h-4c-1.25 0-2 .75-2 1.972V11c0 1.25.75 2 2 2h.75c0 2.25.25 4-2.75 4v3z"/>',
        'play-circle' => '<circle cx="12" cy="12" r="10"/><polygon points="10 8 16 12 10 16 10 8"/>',
        'calendar' => '<rect x="3" y="4" width="18" height="18" rx="2" ry="2"/><line x1="16" y1="2" x2="16" y2="6"/><line x1="8" y1="2" x2="8" y2="6"/><line x1="3" y1="10" x2="21" y2="10"/>',
        'dollar-sign' => '<line x1="12" y1="1" x2="12" y2="23"/><path d="M17 5H9.5a3.5 3.5 0 0 0 0 7h5a3.5 3.5 0 0 1 0 7H6"/>',
        'menu' => '<line x1="3" y1="12" x2="21" y2="12"/><line x1="3" y1="6" x2="21" y2="6"/><line x1="3" y1="18" x2="21" y2="18"/>',
        'x' => '<line x1="18" y1="6" x2="6" y2="18"/><line x1="6" y1="6" x2="18" y2="18"/>',
        'chevron-down' => '<polyline points="6 9 12 15 18 9"/>',
        'facebook' => '<path d="M18 2h-3a5 5 0 0 0-5 5v3H7v4h3v8h4v-8h3l1-4h-4V7a1 1 0 0 1 1-1h3z"/>',
        'instagram' => '<rect x="2" y="2" width="20" height="20" rx="5" ry="5"/><path d="M16 11.37A4 4 0 1 1 12.63 8 4 4 0 0 1 16 11.37z"/><line x1="17.5" y1="6.5" x2="17.51" y2="6.5"/>',
        'youtube' => '<path d="M22.54 6.42a2.78 2.78 0 0 0-1.94-2C18.88 4 12 4 12 4s-6.88 0-8.6.46a2.78 2.78 0 0 0-1.94 2A29 29 0 0 0 1 11.75a29 29 0 0 0 .46 5.33A2.78 2.78 0 0 0 3.4 19.1c1.72.46 8.6.46 8.6.46s6.88 0 8.6-.46a2.78 2.78 0 0 0 1.94-2 29 29 0 0 0 .46-5.25 29 29 0 0 0-.46-5.33z"/><polygon points="9.75 15.02 15.5 11.75 9.75 8.48 9.75 15.02"/>',
        'yelp' => '<path d="M20.16 12.73c-.26-.33-.67-.48-1.05-.48-.19 0-.39.04-.56.13l-2.44 1.22a1 1 0 0 0-.54.89v.01a1 1 0 0 0 .54.89l2.44 1.22c.17.09.37.13.56.13.38 0 .79-.15 1.05-.48.36-.46.36-1.1 0-1.56l-.01-.01.01-.01c.36-.46.36-1.1 0-1.56zM12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2z"/>',
    );

    if ( ! isset( $icons[ $name ] ) ) {
        return '';
    }

    $class_attr = $class ? ' class="ace-icon ' . esc_attr( $class ) . '"' : ' class="ace-icon"';

    return '<svg' . $class_attr . ' width="' . intval( $size ) . '" height="' . intval( $size ) . '" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" xmlns="http://www.w3.org/2000/svg">' . $icons[ $name ] . '</svg>';
}

/**
 * Output SVG icon (echo version)
 */
function ace_the_icon( $name, $size = 24, $class = '' ) {
    echo ace_icon( $name, $size, $class ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- SVG markup is hardcoded
}

/**
 * Get all services data
 */
function ace_get_services() {
    $services = get_option( 'ace_services', array() );
    if ( empty( $services ) && function_exists( 'ace_get_default_options' ) ) {
        $defaults = ace_get_default_options();
        $services = isset( $defaults['ace_services'] ) ? $defaults['ace_services'] : array();
    }
    return $services;
}

/**
 * Get single service by slug
 */
function ace_get_service( $slug ) {
    $services = ace_get_services();
    foreach ( $services as $service ) {
        if ( isset( $service['slug'] ) && $service['slug'] === $slug ) {
            return $service;
        }
    }
    return false;
}

/**
 * Get testimonials
 */
function ace_get_testimonials() {
    $testimonials = get_option( 'ace_testimonials', array() );
    if ( empty( $testimonials ) && function_exists( 'ace_get_default_options' ) ) {
        $defaults = ace_get_default_options();
        $testimonials = isset( $defaults['ace_testimonials'] ) ? $defaults['ace_testimonials'] : array();
    }
    return $testimonials;
}

/**
 * Get stats data
 */
function ace_get_stats() {
    $stats = get_option( 'ace_stats', array() );
    if ( empty( $stats ) && function_exists( 'ace_get_default_options' ) ) {
        $defaults = ace_get_default_options();
        $stats = isset( $defaults['ace_stats'] ) ? $defaults['ace_stats'] : array();
    }
    return $stats;
}

/**
 * Get process steps
 */
function ace_get_process_steps() {
    $steps = get_option( 'ace_process_steps', array() );
    if ( empty( $steps ) && function_exists( 'ace_get_default_options' ) ) {
        $defaults = ace_get_default_options();
        $steps = isset( $defaults['ace_process_steps'] ) ? $defaults['ace_process_steps'] : array();
    }
    return $steps;
}

/**
 * Get Why ACE features
 */
function ace_get_why_ace_features() {
    $features = get_option( 'ace_why_ace_features', array() );
    if ( empty( $features ) && function_exists( 'ace_get_default_options' ) ) {
        $defaults = ace_get_default_options();
        $features = isset( $defaults['ace_why_ace_features'] ) ? $defaults['ace_why_ace_features'] : array();
    }
    return $features;
}

/**
 * Check if a section is visible
 */
function ace_is_section_visible( $section_key ) {
    $sections = get_option( 'ace_sections', array() );
    return ! isset( $sections[ $section_key ] ) || $sections[ $section_key ];
}

/**
 * Get reading time estimate for a post
 */
function ace_reading_time( $post_id = null ) {
    $post = get_post( $post_id );
    if ( ! $post ) {
        return '1 min read';
    }
    $word_count = str_word_count( wp_strip_all_tags( $post->post_content ) );
    $minutes = max( 1, ceil( $word_count / 250 ) );
    return $minutes . ' min read';
}

/**
 * Truncate text to a certain number of words
 */
function ace_truncate( $text, $words = 20, $more = '...' ) {
    return wp_trim_words( $text, $words, $more );
}

/**
 * Generate breadcrumbs
 */
function ace_breadcrumbs() {
    if ( is_front_page() ) {
        return;
    }

    $items = array();
    $items[] = '<a href="' . esc_url( home_url( '/' ) ) . '">' . esc_html__( 'Home', 'ace-theme' ) . '</a>';

    if ( is_singular( 'ace_project' ) ) {
        $items[] = '<a href="' . esc_url( home_url( '/our-work/' ) ) . '">' . esc_html__( 'Our Work', 'ace-theme' ) . '</a>';
        $items[] = '<span>' . esc_html( get_the_title() ) . '</span>';
    } elseif ( is_singular( 'post' ) ) {
        $items[] = '<a href="' . esc_url( get_permalink( get_option( 'page_for_posts' ) ) ) . '">' . esc_html__( 'Blog', 'ace-theme' ) . '</a>';
        $items[] = '<span>' . esc_html( get_the_title() ) . '</span>';
    } elseif ( is_page() ) {
        $ancestors = get_post_ancestors( get_the_ID() );
        foreach ( array_reverse( $ancestors ) as $ancestor ) {
            $items[] = '<a href="' . esc_url( get_permalink( $ancestor ) ) . '">' . esc_html( get_the_title( $ancestor ) ) . '</a>';
        }
        $items[] = '<span>' . esc_html( get_the_title() ) . '</span>';
    } elseif ( is_archive() ) {
        $items[] = '<span>' . esc_html( get_the_archive_title() ) . '</span>';
    } elseif ( is_search() ) {
        $items[] = '<span>' . esc_html__( 'Search Results', 'ace-theme' ) . '</span>';
    } elseif ( is_404() ) {
        $items[] = '<span>' . esc_html__( 'Page Not Found', 'ace-theme' ) . '</span>';
    }

    echo '<nav class="breadcrumbs" aria-label="' . esc_attr__( 'Breadcrumb', 'ace-theme' ) . '">';
    echo '<div class="container">';
    echo wp_kses_post( implode( ' <span class="separator">/</span> ', $items ) );
    echo '</div>';
    echo '</nav>';
}

/**
 * Get service areas
 */
function ace_get_service_areas() {
    $areas = get_option( 'ace_service_areas', array() );
    if ( empty( $areas ) && function_exists( 'ace_get_default_options' ) ) {
        $defaults = ace_get_default_options();
        $areas = isset( $defaults['ace_service_areas'] ) ? $defaults['ace_service_areas'] : array();
    }
    return $areas;
}

/**
 * Get about values
 */
function ace_get_about_values() {
    $values = get_option( 'ace_about_values', array() );
    if ( empty( $values ) && function_exists( 'ace_get_default_options' ) ) {
        $defaults = ace_get_default_options();
        $values = isset( $defaults['ace_about_values'] ) ? $defaults['ace_about_values'] : array();
    }
    return $values;
}

/**
 * Get hero stats for a specific service page.
 *
 * Returns 3 stat items tailored to each service slug.
 */
function ace_get_service_hero_stats( $slug ) {
    $stats = array(
        'full-home-remodeling'  => array(
            array( 'number' => '500+', 'label' => 'Homes remodeled' ),
            array( 'number' => '4.9', 'label' => 'Google rating' ),
            array( 'number' => '15+', 'label' => 'Years in LA' ),
        ),
        'adu'                   => array(
            array( 'number' => '75+', 'label' => 'ADUs completed' ),
            array( 'number' => '100%', 'label' => 'Permit success' ),
            array( 'number' => '15+', 'label' => 'Years in LA' ),
        ),
        'home-additions'        => array(
            array( 'number' => '200+', 'label' => 'Additions built' ),
            array( 'number' => '4.9', 'label' => 'Google rating' ),
            array( 'number' => '15+', 'label' => 'Years in LA' ),
        ),
        'bathroom-remodeling'   => array(
            array( 'number' => '350+', 'label' => 'Bathrooms remodeled' ),
            array( 'number' => '4.9', 'label' => 'Google rating' ),
            array( 'number' => '15+', 'label' => 'Years in LA' ),
        ),
        'kitchen-remodeling'    => array(
            array( 'number' => '300+', 'label' => 'Kitchens remodeled' ),
            array( 'number' => '4.9', 'label' => 'Google rating' ),
            array( 'number' => '15+', 'label' => 'Years in LA' ),
        ),
        'landscape-and-hardscaping' => array(
            array( 'number' => '200+', 'label' => 'Outdoor projects' ),
            array( 'number' => '4.9', 'label' => 'Google rating' ),
            array( 'number' => '15+', 'label' => 'Years in LA' ),
        ),
        'backyard-remodeling'   => array(
            array( 'number' => '250+', 'label' => 'Backyards transformed' ),
            array( 'number' => '4.9', 'label' => 'Google rating' ),
            array( 'number' => '15+', 'label' => 'Years in LA' ),
        ),
        'concrete'              => array(
            array( 'number' => '400+', 'label' => 'Concrete projects' ),
            array( 'number' => '4.9', 'label' => 'Google rating' ),
            array( 'number' => '15+', 'label' => 'Years in LA' ),
        ),
        'driveways'             => array(
            array( 'number' => '300+', 'label' => 'Driveways completed' ),
            array( 'number' => '4.9', 'label' => 'Google rating' ),
            array( 'number' => '15+', 'label' => 'Years in LA' ),
        ),
        'front-yard-renovation' => array(
            array( 'number' => '150+', 'label' => 'Front yards renovated' ),
            array( 'number' => '4.9', 'label' => 'Google rating' ),
            array( 'number' => '15+', 'label' => 'Years in LA' ),
        ),
        'swimming-pools'        => array(
            array( 'number' => '100+', 'label' => 'Pools built' ),
            array( 'number' => '4.9', 'label' => 'Google rating' ),
            array( 'number' => '15+', 'label' => 'Years in LA' ),
        ),
        'outdoor-kitchens'      => array(
            array( 'number' => '120+', 'label' => 'Outdoor kitchens built' ),
            array( 'number' => '4.9', 'label' => 'Google rating' ),
            array( 'number' => '15+', 'label' => 'Years in LA' ),
        ),
        'patio-covers'          => array(
            array( 'number' => '200+', 'label' => 'Patios covered' ),
            array( 'number' => '4.9', 'label' => 'Google rating' ),
            array( 'number' => '15+', 'label' => 'Years in LA' ),
        ),
        'pavers'                => array(
            array( 'number' => '350+', 'label' => 'Paver projects' ),
            array( 'number' => '4.9', 'label' => 'Google rating' ),
            array( 'number' => '15+', 'label' => 'Years in LA' ),
        ),
        'artificial-turf'       => array(
            array( 'number' => '180+', 'label' => 'Turf installations' ),
            array( 'number' => '4.9', 'label' => 'Google rating' ),
            array( 'number' => '15+', 'label' => 'Years in LA' ),
        ),
    );

    if ( isset( $stats[ $slug ] ) ) {
        return $stats[ $slug ];
    }

    return array(
        array( 'number' => '2,000+', 'label' => 'Projects completed' ),
        array( 'number' => '4.9', 'label' => 'Google rating' ),
        array( 'number' => '15+', 'label' => 'Years in LA' ),
    );
}

/**
 * Get Font Awesome icons for hero benefits per service.
 *
 * Returns an array of 4 FA icon class names for the first 4 whats_included items.
 */
function ace_get_service_benefit_icons( $slug ) {
    $icons = array(
        'kitchen-remodeling'        => array( 'fa-drafting-compass', 'fa-cube', 'fa-expand-arrows-alt', 'fa-hard-hat' ),
        'bathroom-remodeling'       => array( 'fa-shower', 'fa-palette', 'fa-lightbulb', 'fa-tools' ),
        'adu'                       => array( 'fa-building', 'fa-file-contract', 'fa-plug', 'fa-door-open' ),
        'landscape-and-hardscaping' => array( 'fa-seedling', 'fa-tint', 'fa-tree', 'fa-sun' ),
        'full-home-remodeling'      => array( 'fa-home', 'fa-ruler-combined', 'fa-solar-panel', 'fa-key' ),
        'concrete'                  => array( 'fa-layer-group', 'fa-stamp', 'fa-shapes', 'fa-warehouse' ),
        'home-additions'            => array( 'fa-expand-arrows-alt', 'fa-drafting-compass', 'fa-hard-hat', 'fa-home' ),
        'backyard-remodeling'       => array( 'fa-tree', 'fa-fire', 'fa-water', 'fa-umbrella-beach' ),
        'driveways'                 => array( 'fa-road', 'fa-truck', 'fa-paint-roller', 'fa-ruler' ),
        'front-yard-renovation'     => array( 'fa-seedling', 'fa-paint-roller', 'fa-lightbulb', 'fa-home' ),
        'swimming-pools'            => array( 'fa-swimming-pool', 'fa-water', 'fa-sun', 'fa-shield-alt' ),
        'outdoor-kitchens'          => array( 'fa-utensils', 'fa-fire', 'fa-couch', 'fa-lightbulb' ),
        'patio-covers'              => array( 'fa-umbrella-beach', 'fa-home', 'fa-lightbulb', 'fa-ruler-combined' ),
        'pavers'                    => array( 'fa-th-large', 'fa-paint-roller', 'fa-road', 'fa-ruler' ),
        'artificial-turf'           => array( 'fa-seedling', 'fa-tint-slash', 'fa-child', 'fa-dog' ),
    );

    if ( isset( $icons[ $slug ] ) ) {
        return $icons[ $slug ];
    }

    return array( 'fa-check-circle', 'fa-check-circle', 'fa-check-circle', 'fa-check-circle' );
}
