<?php
/**
 * FRESH Theme Functions
 *
 * @package Fresh
 * @version 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) exit;

define( 'FRESH_VERSION', '1.0.0' );
define( 'FRESH_DIR', get_template_directory() );
define( 'FRESH_URI', get_template_directory_uri() );

require_once FRESH_DIR . '/inc/icons.php';

/* ==========================================================================
   PLUGIN BRIDGE — fm() helper for reading plugin options
   If FRESH Manager plugin is NOT active, falls back to default values
   ========================================================================== */
if ( ! function_exists( 'fm' ) ) {
    function fm( $group, $key, $default = '' ) { return $default; }
}
if ( ! function_exists( 'freshm_get_all' ) ) {
    function freshm_get_all( $group ) { return array(); }
}
if ( ! function_exists( 'freshm_get_services' ) ) {
    function freshm_get_services( $status = 'publish' ) { return array(); }
}
if ( ! function_exists( 'freshm_get_service_by_slug' ) ) {
    function freshm_get_service_by_slug( $slug ) { return null; }
}
if ( ! function_exists( 'freshm_get_service_cats' ) ) {
    function freshm_get_service_cats() { return array(); }
}
if ( ! function_exists( 'freshm_get_stories' ) ) {
    function freshm_get_stories( $status = 'publish' ) { return array(); }
}
if ( ! function_exists( 'freshm_get_story_by_slug' ) ) {
    function freshm_get_story_by_slug( $slug ) { return null; }
}
if ( ! function_exists( 'freshm_get_team_members' ) ) {
    function freshm_get_team_members() { return array(); }
}
if ( ! function_exists( 'freshm_get_category_page_data' ) ) {
    function freshm_get_category_page_data( $slug ) { return array(); }
}
if ( ! function_exists( 'freshm_parse_pipe_meta' ) ) {
    function freshm_parse_pipe_meta( $text ) {
        if ( empty( $text ) ) return array();
        $items = array();
        foreach ( array_filter( array_map( 'trim', explode( "\n", $text ) ) ) as $line ) {
            $items[] = array_map( 'trim', explode( '|', $line ) );
        }
        return $items;
    }
}
if ( ! function_exists( 'freshm_parse_gallery' ) ) {
    function freshm_parse_gallery( $text ) {
        if ( empty( $text ) ) return array();
        return array_filter( array_map( 'trim', explode( "\n", $text ) ) );
    }
}
if ( ! function_exists( 'freshm_phone' ) ) {
    function freshm_phone( $linked = true ) {
        $phone = fm( 'global', 'phone', '050-123-4567' );
        if ( $linked ) return '<a href="tel:+' . preg_replace( '/[^0-9]/', '', $phone ) . '">' . esc_html( $phone ) . '</a>';
        return esc_html( $phone );
    }
}
if ( ! function_exists( 'freshm_whatsapp_url' ) ) {
    function freshm_whatsapp_url() {
        return 'https://wa.me/' . preg_replace( '/[^0-9]/', '', fm( 'global', 'whatsapp', '972501234567' ) );
    }
}
if ( ! function_exists( 'freshm_email' ) ) {
    function freshm_email( $linked = true ) {
        $email = fm( 'global', 'email', 'hello@fresh.agency' );
        if ( $linked ) return '<a href="mailto:' . esc_attr( $email ) . '">' . esc_html( $email ) . '</a>';
        return esc_html( $email );
    }
}

/* ==========================================================================
   THEME SETUP
   ========================================================================== */
function fresh_setup() {
    // Language / RTL
    load_theme_textdomain( 'fresh', FRESH_DIR . '/languages' );

    // Core support
    add_theme_support( 'title-tag' );
    add_theme_support( 'post-thumbnails' );
    add_theme_support( 'html5', array( 'search-form', 'comment-form', 'gallery', 'caption', 'style', 'script' ) );
    add_theme_support( 'custom-logo', array(
        'height'      => 90,
        'width'       => 200,
        'flex-height' => true,
        'flex-width'  => true,
    ) );
    add_theme_support( 'automatic-feed-links' );

    // Menus
    register_nav_menus( array(
        'primary'  => __( 'תפריט ראשי', 'fresh' ),
        'footer'   => __( 'תפריט פוטר', 'fresh' ),
    ) );

    // Image sizes
    add_image_size( 'fresh-hero', 1920, 1080, true );
    add_image_size( 'fresh-portfolio', 800, 600, true );
    add_image_size( 'fresh-blog-featured', 1200, 750, true );
    add_image_size( 'fresh-blog', 800, 500, true );
    add_image_size( 'fresh-gallery', 400, 400, true );
    add_image_size( 'fresh-team', 400, 400, true );
}
add_action( 'after_setup_theme', 'fresh_setup' );


/* ==========================================================================
   ELEMENTOR SUPPORT
   ========================================================================== */
function fresh_elementor_support() {
    // Register Elementor theme locations
    if ( did_action( 'elementor/loaded' ) ) {
        add_theme_support( 'header-footer-elementor' );
    }
}
add_action( 'after_setup_theme', 'fresh_elementor_support' );

// Register Elementor locations
function fresh_register_elementor_locations( $elementor_theme_manager ) {
    $elementor_theme_manager->register_location( 'header' );
    $elementor_theme_manager->register_location( 'footer' );
    $elementor_theme_manager->register_location( 'single' );
    $elementor_theme_manager->register_location( 'archive' );
}
add_action( 'elementor/theme/register_locations', 'fresh_register_elementor_locations' );

// Add Elementor-compatible body classes
function fresh_body_classes( $classes ) {
    if ( is_page_template( 'page-templates/canvas.php' ) ) {
        $classes[] = 'fresh-canvas';
    }
    if ( is_page_template( 'page-templates/full-width.php' ) ) {
        $classes[] = 'fresh-full-width';
    }
    if ( is_rtl() ) {
        $classes[] = 'rtl';
    }
    $classes[] = 'fresh-theme';
    return $classes;
}
add_filter( 'body_class', 'fresh_body_classes' );


/* ==========================================================================
   ENQUEUE ASSETS
   ========================================================================== */
function fresh_enqueue() {
    // Google Fonts
    wp_enqueue_style(
        'fresh-fonts',
        'https://fonts.googleapis.com/css2?family=Heebo:wght@300;400;500;700;900&family=Playfair+Display:ital,wght@0,400;0,700;1,400&family=Secular+One&display=swap',
        array(),
        null
    );

    // Theme CSS
    wp_enqueue_style( 'fresh-style', get_stylesheet_uri(), array( 'fresh-fonts' ), FRESH_VERSION );

    // Mega Menu CSS (isolated)
    wp_enqueue_style( 'fresh-mega-menu', FRESH_URI . '/assets/css/mega-menu.css', array( 'fresh-style' ), FRESH_VERSION );

    // Theme JS
    wp_enqueue_script( 'fresh-main', FRESH_URI . '/assets/js/main.js', array(), FRESH_VERSION, true );

    // Mega Menu JS
    wp_enqueue_script( 'fresh-mega-menu', FRESH_URI . '/assets/js/mega-menu.js', array(), FRESH_VERSION, true );

    // Localize
    wp_localize_script( 'fresh-main', 'freshData', array(
        'ajaxUrl'  => admin_url( 'admin-ajax.php' ),
        'themeUrl' => FRESH_URI,
        'nonce'    => wp_create_nonce( 'fresh_nonce' ),
    ) );
}
add_action( 'wp_enqueue_scripts', 'fresh_enqueue' );


/* ==========================================================================
   CUSTOM POST TYPES
   ========================================================================== */
function fresh_register_cpts() {

    // Services
    register_post_type( 'fresh_service', array(
        'labels' => array(
            'name'               => 'שירותים',
            'singular_name'      => 'שירות',
            'add_new'            => 'הוסף שירות',
            'add_new_item'       => 'הוסף שירות חדש',
            'edit_item'          => 'ערוך שירות',
            'new_item'           => 'שירות חדש',
            'view_item'          => 'צפה בשירות',
            'search_items'       => 'חפש שירותים',
            'not_found'          => 'לא נמצאו שירותים',
            'not_found_in_trash' => 'לא נמצאו שירותים בפח',
            'all_items'          => 'כל השירותים',
            'menu_name'          => 'שירותים',
        ),
        'public'       => true,
        'has_archive'  => false,
        'rewrite'      => array( 'slug' => 'service', 'with_front' => false ),
        'supports'     => array( 'title', 'editor', 'thumbnail', 'excerpt', 'custom-fields' ),
        'menu_icon'    => 'dashicons-portfolio',
        'show_in_rest' => true,
    ) );

    // Service Categories
    register_taxonomy( 'service_category', 'fresh_service', array(
        'labels' => array(
            'name'              => 'קטגוריות שירות',
            'singular_name'     => 'קטגוריית שירות',
            'add_new_item'      => 'הוסף קטגוריה חדשה',
            'edit_item'         => 'ערוך קטגוריה',
            'search_items'      => 'חפש קטגוריות',
            'all_items'         => 'כל הקטגוריות',
        ),
        'hierarchical'  => true,
        'rewrite'       => array( 'slug' => 'service-category' ),
        'show_in_rest'  => true,
    ) );

    // Client Success Stories
    register_post_type( 'fresh_case_study', array(
        'labels' => array(
            'name'               => 'הצלחת לקוחות',
            'singular_name'      => 'הצלחת לקוח',
            'add_new'            => 'הוסף הצלחת לקוח',
            'add_new_item'       => 'הוסף הצלחת לקוח חדשה',
            'edit_item'          => 'ערוך הצלחת לקוח',
            'new_item'           => 'הצלחת לקוח חדשה',
            'view_item'          => 'צפה בהצלחת לקוח',
            'search_items'       => 'חפש הצלחת לקוחות',
            'not_found'          => 'לא נמצאו הצלחת לקוחות',
            'not_found_in_trash' => 'לא נמצאו הצלחת לקוחות בפח',
            'all_items'          => 'כל הצלחות הלקוחות',
            'menu_name'          => 'הצלחת לקוחות',
        ),
        'public'       => true,
        'has_archive'  => false,
        'rewrite'      => array( 'slug' => 'client-success', 'with_front' => false ),
        'supports'     => array( 'title', 'editor', 'thumbnail', 'excerpt', 'custom-fields' ),
        'menu_icon'    => 'dashicons-analytics',
        'show_in_rest' => true,
    ) );

    // Case Study Industry taxonomy
    register_taxonomy( 'case_study_industry', 'fresh_case_study', array(
        'labels' => array(
            'name'          => 'תעשיות',
            'singular_name' => 'תעשייה',
        ),
        'hierarchical'  => true,
        'rewrite'       => array( 'slug' => 'industry' ),
        'show_in_rest'  => true,
    ) );

    // Team CPT
    register_post_type( 'fresh_team', array(
        'labels' => array(
            'name'          => 'צוות',
            'singular_name' => 'חבר צוות',
            'add_new'       => 'הוסף חבר צוות',
            'add_new_item'  => 'הוסף חבר צוות חדש',
            'edit_item'     => 'ערוך חבר צוות',
            'all_items'     => 'כל חברי הצוות',
            'menu_name'     => 'צוות',
        ),
        'public'       => true,
        'has_archive'  => false,
        'rewrite'      => array( 'slug' => 'team' ),
        'supports'     => array( 'title', 'editor', 'thumbnail', 'custom-fields' ),
        'menu_icon'    => 'dashicons-groups',
        'show_in_rest' => true,
    ) );
}
add_action( 'init', 'fresh_register_cpts' );

// Auto-flush rewrite rules when CPT config changes
function fresh_maybe_flush_rewrites() {
    $ver = 'fresh_cpt_v3'; // bump when changing slugs/archive settings
    if ( get_option( 'fresh_cpt_rewrite_ver' ) !== $ver ) {
        flush_rewrite_rules();
        update_option( 'fresh_cpt_rewrite_ver', $ver );
    }
}
add_action( 'init', 'fresh_maybe_flush_rewrites', 99 );


/* ==========================================================================
   CUSTOM META FIELDS
   ========================================================================== */
function fresh_register_meta() {
    // Service meta
    register_post_meta( 'fresh_service', 'service_icon', array(
        'show_in_rest' => true,
        'single'       => true,
        'type'         => 'string',
        'default'      => '🎯',
    ) );

    // Case Study meta
    $case_fields = array(
        'client_name', 'industry', 'duration', 'main_result',
        'result_leads', 'result_roi', 'result_engagement',
        'client_quote', 'client_quote_author'
    );
    foreach ( $case_fields as $field ) {
        register_post_meta( 'fresh_case_study', $field, array(
            'show_in_rest' => true,
            'single'       => true,
            'type'         => 'string',
        ) );
    }

    // Team meta
    register_post_meta( 'fresh_team', 'team_role', array(
        'show_in_rest' => true,
        'single'       => true,
        'type'         => 'string',
    ) );
    register_post_meta( 'fresh_team', 'team_bio', array(
        'show_in_rest' => true,
        'single'       => true,
        'type'         => 'string',
    ) );
    register_post_meta( 'fresh_team', 'team_favorite_dish', array(
        'show_in_rest' => true,
        'single'       => true,
        'type'         => 'string',
    ) );
}
add_action( 'init', 'fresh_register_meta' );


/* ==========================================================================
   META BOXES (admin interface for custom fields)
   ========================================================================== */
function fresh_add_meta_boxes() {
    add_meta_box( 'fresh_service_meta', 'פרטי שירות', 'fresh_service_meta_callback', 'fresh_service', 'normal', 'high' );
    add_meta_box( 'fresh_case_study_meta', 'פרטי הצלחת לקוח', 'fresh_case_study_meta_callback', 'fresh_case_study', 'normal', 'high' );
    add_meta_box( 'fresh_team_meta', 'פרטי חבר צוות', 'fresh_team_meta_callback', 'fresh_team', 'normal', 'high' );
}
add_action( 'add_meta_boxes', 'fresh_add_meta_boxes' );

function fresh_service_meta_callback( $post ) {
    wp_nonce_field( 'fresh_meta_nonce', 'fresh_meta_nonce_field' );
    $icon = get_post_meta( $post->ID, 'service_icon', true );
    echo '<p><label><strong>אייקון (אימוג\'י):</strong></label><br>';
    echo '<input type="text" name="service_icon" value="' . esc_attr( $icon ) . '" style="width:100px;font-size:1.5rem;">';
    echo '</p>';
}

function fresh_case_study_meta_callback( $post ) {
    wp_nonce_field( 'fresh_meta_nonce', 'fresh_meta_nonce_field' );
    $fields = array(
        'client_name'         => 'שם לקוח',
        'industry'            => 'תעשייה',
        'duration'            => 'משך (למשל: 6 חודשים)',
        'main_result'         => 'תוצאה ראשית',
        'result_leads'        => 'לידים',
        'result_roi'          => 'ROI',
        'result_engagement'   => 'אנגייג\'מנט',
        'client_quote'        => 'ציטוט לקוח',
        'client_quote_author' => 'כותב הציטוט',
    );
    foreach ( $fields as $key => $label ) {
        $val = get_post_meta( $post->ID, $key, true );
        $tag = ( $key === 'client_quote' ) ? 'textarea' : 'input';
        echo '<p><label><strong>' . esc_html( $label ) . ':</strong></label><br>';
        if ( $tag === 'textarea' ) {
            echo '<textarea name="' . esc_attr( $key ) . '" style="width:100%;min-height:80px;">' . esc_textarea( $val ) . '</textarea>';
        } else {
            echo '<input type="text" name="' . esc_attr( $key ) . '" value="' . esc_attr( $val ) . '" style="width:100%;">';
        }
        echo '</p>';
    }
}

function fresh_team_meta_callback( $post ) {
    wp_nonce_field( 'fresh_meta_nonce', 'fresh_meta_nonce_field' );
    $role = get_post_meta( $post->ID, 'team_role', true );
    $bio  = get_post_meta( $post->ID, 'team_bio', true );
    $dish = get_post_meta( $post->ID, 'team_favorite_dish', true );
    echo '<p><label><strong>תפקיד:</strong></label><br>';
    echo '<input type="text" name="team_role" value="' . esc_attr( $role ) . '" style="width:100%;">';
    echo '</p>';
    echo '<p><label><strong>ביוגרפיה:</strong></label><br>';
    echo '<textarea name="team_bio" style="width:100%;min-height:120px;">' . esc_textarea( $bio ) . '</textarea>';
    echo '</p>';
    echo '<p><label><strong>המנה האהובה:</strong></label><br>';
    echo '<input type="text" name="team_favorite_dish" value="' . esc_attr( $dish ) . '" style="width:100%;">';
    echo '</p>';
}

function fresh_save_meta( $post_id ) {
    if ( ! isset( $_POST['fresh_meta_nonce_field'] ) || ! wp_verify_nonce( $_POST['fresh_meta_nonce_field'], 'fresh_meta_nonce' ) ) return;
    if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) return;
    if ( ! current_user_can( 'edit_post', $post_id ) ) return;

    $meta_keys = array( 'service_icon', 'client_name', 'industry', 'duration', 'main_result', 'result_leads', 'result_roi', 'result_revenue', 'result_engagement', 'client_quote', 'client_quote_author', 'team_role', 'team_bio', 'team_favorite_dish' );
    $textarea_keys = array( 'team_bio', 'client_quote' );
    foreach ( $meta_keys as $key ) {
        if ( isset( $_POST[ $key ] ) ) {
            $sanitizer = in_array( $key, $textarea_keys, true ) ? 'sanitize_textarea_field' : 'sanitize_text_field';
            update_post_meta( $post_id, $key, $sanitizer( $_POST[ $key ] ) );
        }
    }
}
add_action( 'save_post', 'fresh_save_meta' );


/* ==========================================================================
   WIDGETS
   ========================================================================== */
function fresh_widgets_init() {
    register_sidebar( array(
        'name'          => 'סיידבר בלוג',
        'id'            => 'blog-sidebar',
        'before_widget' => '<div class="sidebar-card widget %2$s">',
        'after_widget'  => '</div>',
        'before_title'  => '<h3>',
        'after_title'   => '</h3>',
    ) );
}
add_action( 'widgets_init', 'fresh_widgets_init' );


/* ==========================================================================
   HELPERS
   ========================================================================== */

/**
 * Get logo URL
 */
function fresh_logo_url() {
    // 1. Check FRESH Manager plugin setting
    $plugin_logo = fm( 'global', 'logo', '' );
    if ( $plugin_logo ) {
        return $plugin_logo;
    }
    // 2. Check WordPress Customizer
    $custom_logo_id = get_theme_mod( 'custom_logo' );
    if ( $custom_logo_id ) {
        return wp_get_attachment_image_url( $custom_logo_id, 'full' );
    }
    // 3. Fallback to static file
    return FRESH_URI . '/assets/images/logo.png';
}

/**
 * Extract YouTube video ID from various URL formats
 */
function fresh_youtube_id( $url ) {
    $pattern = '/(?:youtube\.com\/(?:watch\?v=|shorts\/|embed\/|v\/)|youtu\.be\/)([a-zA-Z0-9_-]{11})/';
    if ( preg_match( $pattern, $url, $m ) ) {
        return $m[1];
    }
    return '';
}

/**
 * Build embed URL from any video URL (YouTube or Vimeo)
 */
function fresh_embed_url( $url ) {
    $yt_id = fresh_youtube_id( $url );
    if ( $yt_id ) {
        return 'https://www.youtube.com/embed/' . $yt_id . '?autoplay=1&rel=0';
    }
    if ( preg_match( '/vimeo\.com\/(\d+)/', $url, $m ) ) {
        return 'https://player.vimeo.com/video/' . $m[1] . '?autoplay=1';
    }
    return '';
}

/**
 * Get YouTube thumbnail from URL
 */
function fresh_youtube_thumb( $url ) {
    $yt_id = fresh_youtube_id( $url );
    if ( $yt_id ) {
        return 'https://img.youtube.com/vi/' . $yt_id . '/hqdefault.jpg';
    }
    return '';
}

/**
 * Breadcrumb
 */
function fresh_breadcrumb() {
    echo '<div class="breadcrumb">';
    echo '<a href="' . esc_url( home_url() ) . '">ראשי</a>';
    echo '<span class="separator">›</span>';

    if ( is_singular( 'fresh_service' ) ) {
        echo '<a href="' . esc_url( home_url( '/services/' ) ) . '">שירותים</a>';
        echo '<span class="separator">›</span>';
        echo '<span>' . esc_html( get_the_title() ) . '</span>';
    } elseif ( is_singular( 'fresh_case_study' ) ) {
        echo '<a href="' . esc_url( home_url( '/portfolio/' ) ) . '">הצלחת לקוחות</a>';
        echo '<span class="separator">›</span>';
        echo '<span>' . esc_html( get_the_title() ) . '</span>';
    } elseif ( is_singular( 'post' ) ) {
        $blog_page_id = get_option( 'page_for_posts' );
        if ( $blog_page_id ) {
            echo '<a href="' . esc_url( get_permalink( $blog_page_id ) ) . '">בלוג</a>';
        }
        echo '<span class="separator">›</span>';
        echo '<span>' . esc_html( get_the_title() ) . '</span>';
    } elseif ( is_home() ) {
        echo '<span>בלוג</span>';
    } elseif ( is_page() ) {
        $tpl = get_page_template_slug();
        if ( $tpl === 'page-portfolio.php' ) {
            echo '<span>הצלחת לקוחות</span>';
        } else {
            echo '<span>' . esc_html( get_the_title() ) . '</span>';
        }
    } elseif ( is_archive() ) {
        echo '<span>' . esc_html( get_the_archive_title() ) . '</span>';
    } elseif ( is_search() ) {
        echo '<span>חיפוש: ' . esc_html( get_search_query() ) . '</span>';
    }

    echo '</div>';
}

/**
 * Reading time estimate
 */
function fresh_reading_time( $post_id = null ) {
    $post_id = $post_id ?: get_the_ID();
    $content = get_post_field( 'post_content', $post_id );
    $word_count = mb_str_word_count( strip_tags( $content ) );
    $minutes = max( 1, ceil( $word_count / 200 ) );
    return $minutes . ' דק\' קריאה';
}

/**
 * Hebrew-aware word count
 */
if ( ! function_exists( 'mb_str_word_count' ) ) {
    function mb_str_word_count( $string ) {
        return count( preg_split( '/\s+/u', trim( $string ), -1, PREG_SPLIT_NO_EMPTY ) );
    }
}


/* ==========================================================================
   EXCERPT
   ========================================================================== */
function fresh_excerpt_length( $length ) {
    return 25;
}
add_filter( 'excerpt_length', 'fresh_excerpt_length' );

function fresh_excerpt_more( $more ) {
    return '...';
}
add_filter( 'excerpt_more', 'fresh_excerpt_more' );


/* ==========================================================================
   MEGA MENU — Services dropdown from plugin data
   ========================================================================== */
function fresh_services_mega_html() {
    $svcs = freshm_get_services();

    // Index services by slug for fast lookup
    $by_slug = array();
    if ( ! empty( $svcs ) ) {
        foreach ( $svcs as $s ) {
            $by_slug[ $s['slug'] ] = $s;
        }
    }

    // 5 mega-menu categories - 15 services total
    // Each service: slug, name, desc, url (optional override)
    $mega_cats = array(
        array(
            'name'     => 'פרסום ממומן',
            'services' => array(
                array( 'slug' => 'paid-campaigns',       'name' => 'קמפיינים ממומנים',             'desc' => 'Facebook, Instagram, Google, TikTok - מיקוד גיאוגרפי ואופטימיזציה' ),
                array( 'slug' => 'influencer-marketing',  'name' => 'שיווק משפיענים ויוצרי תוכן',  'desc' => 'בלוגרים, טיקטוקיסטים, יוצרי תוכן - תוצאות מדידות' ),
            ),
        ),
        array(
            'name'     => 'סושיאל ותוכן',
            'services' => array(
                array( 'slug' => 'social-management',  'name' => 'ניהול רשתות חברתיות',    'desc' => 'תוכן, קהילה, אנגייג\'מנט - נוכחות שמביאה סועדים' ),
                array( 'slug' => 'food-photography',   'name' => 'צילום מנות',              'desc' => 'צילום אוכל מקצועי שהופך כל מנה לתמונה שמוכרת' ),
                array( 'slug' => 'stills-photography', 'name' => 'צילומי סטילס',            'desc' => 'צילומי אווירה, מרחב וזהות שמספרים את הסיפור של המקום' ),
                array( 'slug' => 'video-production',   'name' => 'צילומי וידאו',             'desc' => 'רילסים, סרטונים ותוכן שגורם לאנשים לעצור לגלול' ),
                array( 'slug' => 'reputation',         'name' => 'ניהול מוניטין וביקורות',   'desc' => 'Google, TripAdvisor - מענה מקצועי ואסטרטגיית ביקורות' ),
            ),
        ),
        array(
            'name'     => 'מיתוג ונוכחות',
            'services' => array(
                array( 'slug' => 'branding',    'name' => 'מיתוג מסעדות',         'desc' => 'לוגו, זהות חזותית, שפה עיצובית וקונספט' ),
                array( 'slug' => 'menu-design', 'name' => 'עיצוב תפריט דיגיטלי', 'desc' => 'QR, דפוס, פלטפורמות - Menu Engineering' ),
                array( 'slug' => 'website',     'name' => 'בניית אתר למסעדה',    'desc' => 'עיצוב, פיתוח, SEO מובנה - מובייל-פירסט' ),
            ),
        ),
        array(
            'name'     => 'ניהול פלטפורמות',
            'services' => array(
                array( 'slug' => 'delivery-optimization',    'name' => 'אופטימיזציית משלוחים',      'desc' => 'Wolt, 10bis, Cibus - דירוג, תמונות, רווחיות' ),
                array( 'slug' => 'reservation-optimization', 'name' => 'אופטימיזציית הזמנות שולחן', 'desc' => 'Tabit, Eatbit, Google - תפוסה מקסימלית, מינימום No-shows' ),
                array( 'slug' => 'loyalty',                  'name' => 'ניהול מועדון לקוחות',       'desc' => 'נקודות, הטבות, WhatsApp - לקוחות שחוזרים שוב ושוב' ),
            ),
        ),
        array(
            'name'     => 'צמיחה ומדידה',
            'services' => array(
                array( 'slug' => 'seo',       'name' => 'קידום אורגני - SEO', 'desc' => 'דירוג ב-Google, Business Profile, חיפוש מקומי' ),
                array( 'slug' => 'analytics', 'name' => 'אנליטיקס ודוחות',    'desc' => 'מדידה מדויקת, דשבורדים, החלטות מבוססות דאטה' ),
            ),
        ),
    );

    ob_start();
    ?>
    <div class="mega-menu" aria-label="שירותים">
      <div class="mega-body">
        <!-- Right panel: category sidebar (first in HTML = right in RTL) -->
        <div class="mega-cats">
          <?php foreach ( $mega_cats as $ci => $mc ) : ?>
          <button class="mega-cat<?php echo $ci === 0 ? ' active' : ''; ?>" data-cat="<?php echo $ci; ?>"><?php echo esc_html( $mc['name'] ); ?></button>
          <?php endforeach; ?>
          <a href="<?php echo esc_url( home_url( '/services/' ) ); ?>" class="mega-all-link">כל השירותים ←</a>
        </div>

        <!-- Left panel: service grids + CTA -->
        <div class="mega-panel">
          <?php foreach ( $mega_cats as $ci => $mc ) :
            $svc_count = count( $mc['services'] );
            $cols_class = $svc_count >= 4 ? 'cols-3' : 'cols-2';
          ?>
          <div class="mega-svc-grid<?php echo $ci === 0 ? ' active' : ''; ?>" data-cat="<?php echo $ci; ?>" data-cat-name="<?php echo esc_attr( $mc['name'] ); ?>">
            <h4 class="mega-cat-heading"><?php echo esc_html( $mc['name'] ); ?></h4>
            <div class="mega-svc-items <?php echo $cols_class; ?>">
            <?php foreach ( $mc['services'] as $svc ) :
              $slug = $svc['slug'];
              $url  = isset( $svc['url'] ) ? home_url( $svc['url'] ) : home_url( '/service/' . $slug . '/' );

              // Render icon: from plugin data if available, else built-in icon registry
              $icon_html = '';
              if ( isset( $by_slug[ $slug ] ) && function_exists( 'freshm_render_service_icon' ) ) {
                  $icon_html = freshm_render_service_icon( $by_slug[ $slug ], 'mini' );
              } elseif ( function_exists( 'freshm_get_service_icon' ) ) {
                  $icon_html = freshm_get_service_icon( $slug, 20 );
              }
              if ( ! $icon_html ) {
                  $icon_html = fresh_svg_icon( '◆', 20 );
              }
            ?>
            <a href="<?php echo esc_url( $url ); ?>" class="mega-svc-item">
              <span class="mega-svc-icon"><?php echo $icon_html; ?></span>
              <span class="mega-svc-info">
                <span class="mega-svc-name"><?php echo esc_html( $svc['name'] ); ?></span>
                <span class="mega-svc-desc"><?php echo esc_html( $svc['desc'] ); ?></span>
              </span>
            </a>
            <?php endforeach; ?>
            </div>
            <div class="mega-cta">
              <div class="mega-cta-content">
                <span class="mega-cta-title">לא בטוחים מה מתאים למסעדה?</span>
                <span class="mega-cta-desc">נשמח לייעץ ולהתאים חבילה - בלי התחייבות</span>
              </div>
              <a href="#" class="btn-primary mega-cta-btn" data-popup="contact">קבעו שיחה ←</a>
            </div>
          </div>
          <?php endforeach; ?>
        </div>
      </div>
    </div>
    <?php
    return ob_get_clean();
}

/** Custom Walker — injects mega menu into the services nav item */
class Fresh_Mega_Walker extends Walker_Nav_Menu {
    function start_el( &$output, $item, $depth = 0, $args = null, $id = 0 ) {
        if ( $depth === 0 && strpos( $item->url, '/services' ) !== false ) {
            $item->classes[] = 'has-mega';
        }
        parent::start_el( $output, $item, $depth, $args, $id );
        if ( $depth === 0 && strpos( $item->url, '/services' ) !== false ) {
            $output .= fresh_services_mega_html();
        }
    }
}

/* ---------- FALLBACK MENU ---------- */
function fresh_fallback_menu() {
    $blog_enabled = fm( 'blog_settings', 'enabled', '1' );
    $mega = fresh_services_mega_html();
    echo '<ul class="nav-links" id="navLinks">';
    echo '<li><a href="' . esc_url( home_url( '/' ) ) . '">ראשי</a></li>';
    echo '<li><a href="' . esc_url( home_url( '/about/' ) ) . '">מי אנחנו</a></li>';
    echo '<li class="' . ( $mega ? 'has-mega' : '' ) . '"><a href="' . esc_url( home_url( '/services/' ) ) . '">שירותים</a>' . $mega . '</li>';
    echo '<li><a href="' . esc_url( home_url( '/#showreel' ) ) . '">גלריית וידאו</a></li>';
    echo '<li><a href="' . esc_url( home_url( '/portfolio/' ) ) . '">הצלחת לקוחות</a></li>';
    if ( $blog_enabled === '1' ) {
        $blog_page_id = get_option( 'page_for_posts' );
        $blog_url = $blog_page_id ? get_permalink( $blog_page_id ) : home_url( '/blog/' );
        echo '<li><a href="' . esc_url( $blog_url ) . '">בלוג</a></li>';
    }
    echo '<li><a href="' . esc_url( home_url( '/faq/' ) ) . '">שאלות ותשובות</a></li>';
    echo '<li class="mobile-only"><a href="' . esc_url( home_url( '/contact/' ) ) . '">צור קשר</a></li>';
    echo '</ul>';
}


/* ---------- HIDE BLOG MENU ITEM WHEN DISABLED ---------- */
add_filter( 'wp_nav_menu_objects', 'fresh_filter_blog_menu_item', 10, 2 );
function fresh_filter_blog_menu_item( $items, $args ) {
    if ( fm( 'blog_settings', 'enabled', '1' ) === '0' ) {
        $blog_page_id = get_option( 'page_for_posts' );
        foreach ( $items as $key => $item ) {
            // Match by object_id (page ID) or by URL containing /blog/
            if ( ( $blog_page_id && (int) $item->object_id === (int) $blog_page_id )
                || strpos( $item->url, '/blog' ) !== false ) {
                unset( $items[ $key ] );
            }
        }
    }
    return $items;
}

/* ==========================================================================
   CONTACT FORM HANDLER (AJAX)
   ========================================================================== */
function fresh_handle_contact() {
    check_ajax_referer( 'fresh_nonce', 'nonce' );

    $name       = sanitize_text_field( $_POST['name'] ?? '' );
    $restaurant = sanitize_text_field( $_POST['restaurant'] ?? '' );
    $email      = sanitize_email( $_POST['email'] ?? '' );
    $phone      = sanitize_text_field( $_POST['phone'] ?? '' );
    $message    = sanitize_textarea_field( $_POST['message'] ?? '' );

    if ( empty( $name ) || empty( $phone ) ) {
        wp_send_json_error( array( 'message' => 'נא למלא את כל השדות הנדרשים.' ) );
    }

    $to = get_option( 'admin_email' );
    $subject = 'פנייה חדשה מאתר FRESH: ' . $name;
    $body = "שם: {$name}\nטלפון: {$phone}";
    if ( $restaurant ) $body .= "\nמסעדה: {$restaurant}";
    if ( $email ) $body .= "\nאימייל: {$email}";
    if ( $message ) $body .= "\n\nהודעה:\n{$message}";
    $headers = array( 'Content-Type: text/plain; charset=UTF-8' );
    if ( $email ) $headers[] = 'Reply-To: ' . $name . ' <' . $email . '>';

    $sent = wp_mail( $to, $subject, $body, $headers );

    if ( $sent ) {
        wp_send_json_success( array( 'message' => 'ההודעה נשלחה בהצלחה! נחזור אליכם בהקדם.' ) );
    } else {
        wp_send_json_error( array( 'message' => 'שגיאה בשליחה. נסו שוב מאוחר יותר.' ) );
    }
}
add_action( 'wp_ajax_fresh_contact', 'fresh_handle_contact' );
add_action( 'wp_ajax_nopriv_fresh_contact', 'fresh_handle_contact' );


/* ==========================================================================
   OPEN GRAPH & SEO META TAGS
   ========================================================================== */
function fresh_og_meta() {
    if ( is_admin() ) return;

    $site_name = get_bloginfo( 'name' );
    $site_desc = get_bloginfo( 'description' );
    $locale    = 'he_IL';

    echo '<meta property="og:locale" content="' . esc_attr( $locale ) . '">' . "\n";
    echo '<meta property="og:site_name" content="' . esc_attr( $site_name ) . '">' . "\n";

    if ( is_singular() ) {
        $title = get_the_title();
        $desc  = has_excerpt() ? get_the_excerpt() : wp_trim_words( get_the_content(), 30, '...' );
        $url   = get_permalink();
        $type  = is_front_page() ? 'website' : 'article';
        $image = has_post_thumbnail() ? get_the_post_thumbnail_url( null, 'large' ) : '';

        echo '<meta property="og:type" content="' . esc_attr( $type ) . '">' . "\n";
        echo '<meta property="og:title" content="' . esc_attr( $title ) . '">' . "\n";
        echo '<meta property="og:description" content="' . esc_attr( $desc ) . '">' . "\n";
        echo '<meta property="og:url" content="' . esc_url( $url ) . '">' . "\n";
        if ( $image ) {
            echo '<meta property="og:image" content="' . esc_url( $image ) . '">' . "\n";
        }

        // Twitter Card
        echo '<meta name="twitter:card" content="summary_large_image">' . "\n";
        echo '<meta name="twitter:title" content="' . esc_attr( $title ) . '">' . "\n";
        echo '<meta name="twitter:description" content="' . esc_attr( $desc ) . '">' . "\n";
        if ( $image ) {
            echo '<meta name="twitter:image" content="' . esc_url( $image ) . '">' . "\n";
        }
    } else {
        echo '<meta property="og:type" content="website">' . "\n";
        echo '<meta property="og:title" content="' . esc_attr( $site_name ) . '">' . "\n";
        echo '<meta property="og:description" content="' . esc_attr( $site_desc ) . '">' . "\n";
        echo '<meta property="og:url" content="' . esc_url( home_url( '/' ) ) . '">' . "\n";
    }
}
add_action( 'wp_head', 'fresh_og_meta', 5 );


/* ==========================================================================
   SCHEMA.ORG — LocalBusiness JSON-LD
   ========================================================================== */
function fresh_schema_jsonld() {
    if ( ! is_front_page() ) return;

    $name    = fm( 'global', 'business_name', get_bloginfo( 'name' ) );
    $phone   = fm( 'global', 'phone', '050-123-4567' );
    $email   = fm( 'global', 'email', 'hello@fresh.agency' );
    $address = fm( 'global', 'address', 'רוטשילד 45, תל אביב' );
    $hours   = fm( 'global', 'hours', 'א׳-ה׳ 09:00-18:00' );
    $logo    = fresh_logo_url();

    $schema = array(
        '@context'    => 'https://schema.org',
        '@type'       => 'LocalBusiness',
        'name'        => $name,
        'image'       => $logo,
        'telephone'   => $phone,
        'email'       => $email,
        'address'     => array(
            '@type'           => 'PostalAddress',
            'streetAddress'   => $address,
            'addressLocality' => 'Tel Aviv',
            'addressCountry'  => 'IL',
        ),
        'url'         => home_url( '/' ),
        'openingHours' => $hours,
        'priceRange'   => '$$',
    );

    echo '<script type="application/ld+json">' . wp_json_encode( $schema, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT ) . '</script>' . "\n";
}
add_action( 'wp_head', 'fresh_schema_jsonld', 10 );


/* ==========================================================================
   PERFORMANCE — Lazy loading, Preconnect, Defer
   ========================================================================== */
function fresh_add_preconnect() {
    echo '<link rel="preconnect" href="https://fonts.googleapis.com" crossorigin>' . "\n";
    echo '<link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>' . "\n";
}
add_action( 'wp_head', 'fresh_add_preconnect', 1 );

function fresh_defer_scripts( $tag, $handle, $src ) {
    // Don't defer in admin or if already has defer
    if ( is_admin() || strpos( $tag, 'defer' ) !== false ) return $tag;

    // Defer main theme JS
    if ( $handle === 'fresh-main' ) {
        return str_replace( ' src', ' defer src', $tag );
    }
    return $tag;
}
add_filter( 'script_loader_tag', 'fresh_defer_scripts', 10, 3 );


/* ==========================================================================
   THEME ACTIVATION — AUTO-CREATE PAGES, DEMO CONTENT & SETTINGS
   ========================================================================== */
function fresh_activation() {
    fresh_register_cpts();
    flush_rewrite_rules();

    if ( get_option( 'fresh_demo_installed' ) ) return;

    // --- PAGES ---
    $pages_data = array(
        'home'      => 'דף הבית',
        'about'     => 'מי אנחנו',
        'services'  => 'שירותים',
        'portfolio' => 'הצלחת לקוחות',
        'contact'   => 'צור קשר',
        'blog'      => 'בלוג',
    );
    $page_ids = array();
    foreach ( $pages_data as $slug => $title ) {
        $existing = get_page_by_path( $slug );
        if ( $existing ) { $page_ids[ $slug ] = $existing->ID; continue; }
        $id = wp_insert_post( array( 'post_type' => 'page', 'post_title' => $title, 'post_content' => '', 'post_status' => 'publish', 'post_name' => $slug ) );
        if ( ! is_wp_error( $id ) ) $page_ids[ $slug ] = $id;
    }
    if ( isset( $page_ids['home'] ) ) { update_option( 'show_on_front', 'page' ); update_option( 'page_on_front', $page_ids['home'] ); }
    if ( isset( $page_ids['blog'] ) ) { update_option( 'page_for_posts', $page_ids['blog'] ); }

    // --- MENU ---
    $menu_name = 'תפריט ראשי';
    if ( ! wp_get_nav_menu_object( $menu_name ) ) {
        $menu_id = wp_create_nav_menu( $menu_name );
        if ( ! is_wp_error( $menu_id ) ) {
            $items = array( array('ראשי','home',1), array('מי אנחנו','about',2), array('שירותים','services',3), array('הצלחת לקוחות','portfolio',4), array('בלוג','blog',5) );
            foreach ( $items as $i ) {
                if ( isset( $page_ids[ $i[1] ] ) ) {
                    wp_update_nav_menu_item( $menu_id, 0, array( 'menu-item-title' => $i[0], 'menu-item-object-id' => $page_ids[ $i[1] ], 'menu-item-object' => 'page', 'menu-item-type' => 'post_type', 'menu-item-status' => 'publish', 'menu-item-position' => $i[2] ) );
                }
            }
            set_theme_mod( 'nav_menu_locations', array( 'primary' => $menu_id ) );
        }
    }

    // --- SERVICE CATEGORIES (6 categories) ---
    $service_cats = array(
        'paid-advertising'       => 'פרסום ממומן',
        'social-media-content'   => 'סושיאל מדיה וייצור תוכן',
        'branding-identity'      => 'מיתוג וזהות דיגיטלית',
        'delivery-platforms'     => 'ניהול פלטפורמות משלוחים',
        'reservation-systems'    => 'ניהול מערכות הזמנת שולחן',
        'digital-presence-growth'=> 'נוכחות דיגיטלית וצמיחה',
    );
    foreach ( $service_cats as $slug => $name ) { if ( ! term_exists( $slug, 'service_category' ) ) wp_insert_term( $name, 'service_category', array( 'slug' => $slug ) ); }

    // --- DEMO SERVICES (12) — seed via plugin if available ---
    if ( function_exists( 'freshm_seed_default_services' ) ) {
        freshm_seed_default_services( false );
    } else {
        // Minimal fallback: 12 services as CPT posts
        $svc = array(
            array( 'paid-advertising', 'paid-campaigns', 'קמפיינים ממומנים', 'ניהול קמפיינים ב-Facebook, Instagram, Google ו-TikTok עם מיקוד גיאוגרפי ואופטימיזציה יומית.' ),
            array( 'paid-advertising', 'influencer-marketing', 'שיווק באמצעות משפיענים', 'חיבור המסעדה שלכם לבלוגרי אוכל, טיקטוקיסטים ומשפיעני לייפסטייל.' ),
            array( 'social-media-content', 'social-management', 'ניהול רשתות חברתיות', 'ניהול מלא של Instagram, Facebook ו-TikTok - אסטרטגיית תוכן, עיצוב וקהילה.' ),
            array( 'social-media-content', 'photo-video', 'צילום מנות ווידאו', 'צילומי סטילס, סרטוני ריילס ו-TikTok, סרטי תדמית - תוכן שגורם לרעוב.' ),
            array( 'branding-identity', 'branding', 'מיתוג מסעדות', 'בניית זהות מותגית מאפס - לוגו, שפה עיצובית, קונספט וסיפור מותג.' ),
            array( 'branding-identity', 'menu-design', 'עיצוב תפריט דיגיטלי', 'צילום ועיצוב תפריט דיגיטלי עם QR Code, דפוס, והתאמה למשלוחים.' ),
            array( 'branding-identity', 'website', 'בניית אתר למסעדה', 'אתרים מותאמים אישית עם הזמנות, תפריט, גלריה, בלוג ו-SEO.' ),
            array( 'delivery-platforms', 'delivery-optimization', 'אופטימיזציית משלוחים', 'שיפור ביצועים ב-Wolt, 10bis, Cibus ו-Lunchbox - תפריט, תמחור וצילום.' ),
            array( 'reservation-systems', 'reservation-optimization', 'אופטימיזציית הזמנות שולחן', 'ניהול Tabit, Eatbit, Google Reserve - למילוי מושלם כל ערב.' ),
            array( 'digital-presence-growth', 'seo', 'קידום אורגני - SEO', 'דירוג בגוגל מפות ותוצאות חיפוש - Google Business Profile, תוכן וביקורות.' ),
            array( 'digital-presence-growth', 'reputation', 'ניהול מוניטין וביקורות', 'ניהול ביקורות בגוגל, TripAdvisor ורשתות - מענה מקצועי ובניית אמינות.' ),
            array( 'digital-presence-growth', 'loyalty', 'ניהול מועדון לקוחות', 'מועדון לקוחות דיגיטלי - נקודות, הטבות, SMS, WhatsApp ואוטומציות.' ),
        );
        $order = 1;
        foreach ( $svc as $s ) {
            if ( get_page_by_path( $s[1], OBJECT, 'fresh_service' ) ) { $order++; continue; }
            $id = wp_insert_post( array( 'post_type' => 'fresh_service', 'post_title' => $s[2], 'post_content' => '', 'post_excerpt' => $s[3], 'post_status' => 'publish', 'post_name' => $s[1], 'menu_order' => $order ) );
            if ( ! is_wp_error( $id ) ) {
                update_post_meta( $id, 'service_icon', $s[1] );
                $term = get_term_by( 'slug', $s[0], 'service_category' );
                if ( $term ) wp_set_object_terms( $id, $term->term_id, 'service_category' );
            }
            $order++;
        }
    }

    // --- CLIENT SUCCESS INDUSTRIES ---
    $industries = array( 'italian' => 'מסעדה איטלקית', 'sushi' => 'מסעדת סושי', 'vegan' => 'בריאות וטבעוני', 'burger' => 'המבורגריה', 'cafe' => 'בית קפה', 'bakery' => 'קונדיטוריה' );
    foreach ( $industries as $slug => $name ) { if ( ! term_exists( $slug, 'case_study_industry' ) ) wp_insert_term( $name, 'case_study_industry', array( 'slug' => $slug ) ); }

    // --- DEMO CLIENT SUCCESS STORIES (6) ---
    $cases = array(
        array( 'ROMA TLV', 'italian', '<h2>האתגר</h2><p>ROMA TLV, מסעדה איטלקית בלב תל אביב, רצתה להגדיל את ההזמנות האונליין שלה. הנוכחות הדיגיטלית הייתה מינימלית.</p><h2>הפתרון</h2><p>בנינו אסטרטגיה דיגיטלית מקיפה - ניהול סושיאל, קמפיינים ממומנים, אופטימיזציית משלוחים, וצילום מנות מקצועי.</p><h2>התהליך</h2><p>צילום מקצועי של כל התפריט, בניית אסטרטגיית תוכן ל-3 חודשים, הקמת קמפיינים ב-Facebook ו-Instagram, ואופטימיזציה של Wolt ו-10bis.</p>', 'אבי כהן', 'מסעדה איטלקית', '6 חודשים', '+280% בהזמנות אונליין', '+340%', '480%', '+520%', 'מאז שהתחלנו לעבוד עם FRESH, ההכנסות שלנו מהזמנות אונליין עלו ב-280%. הם פשוט מבינים את העולם של מסעדנות.', 'אבי כהן, בעלים ROMA TLV' ),
        array( 'SAKURA', 'sushi', '<h2>האתגר</h2><p>SAKURA, מסעדת סושי פרימיום, רצתה לבנות קהילה באינסטגרם ולהפוך לשם מוביל. הגיעו עם 2,000 עוקבים.</p><h2>הפתרון</h2><p>אסטרטגיית תוכן ויזואלית - צילום מנות, סרטוני הכנה, שיתופי פעולה עם משפיענים, וסטוריז יומיים.</p><h2>התוצאה</h2><p>תוך 3 חודשים, חשבון האינסטגרם קפץ מ-2,000 ל-15,000 עוקבים. ההזמנות מאינסטגרם עלו ב-180%.</p>', 'שרון לוי', 'מסעדת סושי', '3 חודשים', '+450% עוקבים באינסטגרם', '+180%', '350%', '+650%', 'הצוות של FRESH הפך את האינסטגרם שלנו מ-2,000 ל-15,000 עוקבים תוך 3 חודשים. התוכן שהם יוצרים פשוט ברמה אחרת.', 'שרון לוי, מנהלת SAKURA' ),
        array( 'GREEN QUEEN', 'vegan', '<h2>האתגר</h2><p>GREEN QUEEN, רשת מסעדות בריאות, שילמה עמלות גבוהות ורצתה לבנות ערוצי הזמנה ישירים.</p><h2>הפתרון</h2><p>אפליקציית הזמנות, מועדון לקוחות, קמפיינים שהסיטו לקוחות מפלטפורמות חיצוניות להזמנה ישירה.</p><h2>התוצאה</h2><p>חסכנו 40% בעמלות, 8,000 חברי מועדון, ערוץ ישיר שמהווה 45% מסך ההכנסות.</p>', 'דנה ברק', 'בריאות וטבעוני', '12 חודשים', '-40% בעמלות משלוחים', '8,000 חברי מועדון', '620%', '+380%', 'חסכנו 40% בעמלות משלוחים אחרי שעברנו לאסטרטגיה של FRESH. עכשיו יש לנו קהל לקוחות ישיר שמזמין דרכנו.', 'דנה ברק, שותפה GREEN QUEEN' ),
        array( 'BURGER NATION', 'burger', '<h2>האתגר</h2><p>BURGER NATION פתחה סניף חדש ורצתה לייצר באזז ותנועה מהיום הראשון.</p><h2>הפתרון</h2><p>קמפיין השקה מסיבי ב-TikTok, Instagram ו-Google Ads, 10 משפיעני אוכל, ומבצע פתיחה ויראלי.</p><h2>התוצאה</h2><p>תור של שעתיים ביום הפתיחה, 50,000 צפיות בסרטון ההשקה, +180% בהכנסות ביחס לתחזית.</p>', 'רון אברהם', 'המבורגריה', '2 חודשים', '+180% בהכנסות חודשיות', '+250%', '520%', '+800%', 'FRESH הפכו את הפתיחה שלנו לאירוע. תור של שעתיים ביום הראשון - בזכות הקמפיין המטורף שהם בנו.', 'רון אברהם, בעלים BURGER NATION' ),
        array( 'NOIR CAFÉ', 'cafe', '<h2>האתגר</h2><p>NOIR CAFÉ, בית קפה בוטיק, רצה לבלוט מול רשתות הקפה הגדולות ולמשוך קהל איכותי.</p><h2>הפתרון</h2><p>SEO מקומי, Google Business Profile, ותוכן אינסטגרם שמספר את הסיפור הייחודי.</p><h2>התוצאה</h2><p>+320% בלידים מגוגל, +200% בביקורות חיוביות, 45% יותר לקוחות חדשים בחודש.</p>', 'מיכל דוד', 'בית קפה', '4 חודשים', '+320% בלידים מגוגל', '+320%', '410%', '+290%', 'בזכות FRESH, אנחנו מופיעים ראשונים בגוגל על כל חיפוש של בית קפה באזור. ההכנסות שלנו עלו ב-65%.', 'מיכל דוד, בעלים NOIR CAFÉ' ),
        array( 'SWEET SPOT', 'bakery', '<h2>האתגר</h2><p>SWEET SPOT, קונדיטוריה בוטיק, רצתה לפתח ערוץ מכירות אונליין.</p><h2>הפתרון</h2><p>חנות אונליין, מערכת הזמנות מותאמת, קמפיינים ממוקדים לימי הולדת, חגים ואירועים.</p><h2>התוצאה</h2><p>המכירות האונליין מהוות 40% מסך ההכנסות. +200% במכירות תוך 6 חודשים.</p>', 'נועה גולן', 'קונדיטוריה', '6 חודשים', '+200% במכירות אונליין', '+280%', '380%', '+420%', 'לא חשבנו שאפשר למכור עוגות באונליין ברמה כזו. FRESH בנו לנו מערכת שעובדת מצוין.', 'נועה גולן, בעלים SWEET SPOT' ),
    );
    foreach ( $cases as $c ) {
        if ( get_page_by_path( sanitize_title( $c[0] ), OBJECT, 'fresh_case_study' ) ) continue;
        $id = wp_insert_post( array( 'post_type' => 'fresh_case_study', 'post_title' => $c[0], 'post_content' => $c[2], 'post_status' => 'publish' ) );
        if ( ! is_wp_error( $id ) ) {
            update_post_meta( $id, 'client_name', $c[3] );
            update_post_meta( $id, 'industry', $c[4] );
            update_post_meta( $id, 'duration', $c[5] );
            update_post_meta( $id, 'main_result', $c[6] );
            update_post_meta( $id, 'result_leads', $c[7] );
            update_post_meta( $id, 'result_roi', $c[8] );
            update_post_meta( $id, 'result_engagement', $c[9] );
            update_post_meta( $id, 'client_quote', $c[10] );
            update_post_meta( $id, 'client_quote_author', $c[11] );
            $term = get_term_by( 'slug', $c[1], 'case_study_industry' );
            if ( $term ) wp_set_object_terms( $id, $term->term_id, 'case_study_industry' );
        }
    }

    // --- TEAM MEMBERS (6) ---
    $team = array(
        array( 'יוסי כהן', 'מייסד ומנכ"ל', 'יוסי הקים את FRESH אחרי 8 שנים בניהול מסעדות ו-5 שנים בשיווק דיגיטלי.' ),
        array( 'מאיה לוי', 'מנהלת קריאייטיב', 'מאיה אחראית על כל הצד היצירתי - מברנדינג ועיצוב, דרך צילום ווידאו, ועד תוכן שיווקי.' ),
        array( 'עומר דוד', 'ראש מחלקת שיווק', 'עומר מנהל את כל הקמפיינים הממומנים. הוביל קמפיינים בהיקף מיליוני שקלים.' ),
        array( 'נועה ישראלי', 'מנהלת לקוחות', 'נועה מוודאת שכל לקוח מקבל שירות אישי, תקשורת שוטפת, ותוצאות.' ),
        array( 'דני פרידמן', 'ראש מחלקת סושיאל', 'מומחה לטרנדים, ריילס ו-TikTok - יודע מה הולך להיות ויראלי לפני כולם.' ),
        array( 'שירה אלון', 'צלמת ראשית', 'שירה מצלמת אוכל כבר 6 שנים. התמונות שלה מופיעות במגזינים ובסושיאל של הלקוחות.' ),
    );
    foreach ( $team as $m ) {
        if ( get_page_by_path( sanitize_title( $m[0] ), OBJECT, 'fresh_team' ) ) continue;
        $id = wp_insert_post( array( 'post_type' => 'fresh_team', 'post_title' => $m[0], 'post_content' => $m[2], 'post_status' => 'publish' ) );
        if ( ! is_wp_error( $id ) ) update_post_meta( $id, 'team_role', $m[1] );
    }

    // --- BLOG POSTS (3) ---
    $posts = array(
        array( '10 טעויות שיווק שמסעדות עושות (ואיך להימנע מהן)', 'רוב המסעדות בישראל עושות את אותן טעויות בשיווק דיגיטלי.', '<p>רוב המסעדות בישראל משקיעות כסף בשיווק דיגיטלי, אבל לא תמיד בצורה הנכונה.</p><h2>1. להתמקד בעוקבים במקום בלקוחות</h2><p>הרבה מסעדות רודפות אחרי מספרי עוקבים, אבל מה שבאמת חשוב הוא כמה מהעוקבים הופכים ללקוחות משלמים.</p><h2>2. לפרסם בלי אסטרטגיה</h2><p>פרסום ספונטני הוא לא אסטרטגיה. צריך תוכנית תוכן חודשית מתוכננת מראש.</p><h2>3. להזניח את Google Business Profile</h2><p>84% מהלקוחות מחפשים מסעדות בגוגל. פרופיל לא מעודכן = לקוחות אבודים.</p><h2>4. לא להשקיע בצילום מקצועי</h2><p>תמונות גרועות מהטלפון לא ימכרו את המנות. השקעה בצילום מחזירה את עצמה תוך שבועות.</p><h2>5. לשלם עמלות מיותרות</h2><p>אם 100% מההזמנות עוברות דרך Wolt ו-10bis - אתם משלמים 25-35% עמלה. בניית ערוצים ישירים חוסכת מאות אלפי שקלים.</p>', array('שיווק דיגיטלי'), array('טיפים','שיווק','מסעדות') ),
        array( 'איך TikTok משנה את עולם המסעדנות ב-2025', 'TikTok הפך למנוע גילוי מסעדות #1.', '<p>TikTok הפך לכוח דומיננטי בעולם גילוי המסעדות. 35% מהמשתמשים גילו מסעדה חדשה דרך הפלטפורמה.</p><h2>המספרים מדברים</h2><p>בשנת 2025, האשטאג #FoodTok צבר מעל 80 מיליארד צפיות. מסעדות שמפרסמות ב-TikTok רואות עלייה של 150% בביקורים חדשים.</p><h2>מה עובד?</h2><p>הסוד הוא אותנטיות. לקוחות רוצים לראות את המטבח מאחורי הקלעים, את השף בפעולה, ואת התהליך.</p><h2>טרנדים</h2><ul><li>סרטוני "מאחורי הקלעים"</li><li>הכנת מנה מ-A עד Z</li><li>ASMR של בישול</li><li>אתגרי אוכל</li></ul>', array('סושיאל'), array('TikTok','סושיאל','טרנדים','וידאו') ),
        array( 'מדריך: בניית מועדון לקוחות למסעדה', 'מדריך מעשי לבניית מועדון לקוחות שמגדיל תדירות ביקורים.', '<p>מועדון לקוחות הוא אחד הכלים האפקטיביים ביותר לשימור לקוחות.</p><h2>למה מועדון לקוחות?</h2><p>לקוח חוזר שווה פי 5 מלקוח חדש. מועדון מגדיל תדירות ביקורים ב-35% בממוצע.</p><h2>שלב 1: בחירת פלטפורמה</h2><p>WhatsApp Business + קופונים דיגיטליים - הדרך המשתלמת ביותר.</p><h2>שלב 2: תכנון הטבות</h2><ul><li>קינוח חינם בביקור הראשון</li><li>הנחה של 10% מכל הזמנה חמישית</li><li>מנה חינם ביום ההולדת</li></ul><h2>שלב 3: גיוס חברים</h2><p>המפתח הוא לגייס בנקודת המכירה - כשהלקוח כבר שם ומרוצה.</p>', array('CRM'), array('CRM','מועדון לקוחות','מדריך') ),
    );
    foreach ( $posts as $p ) {
        if ( get_page_by_path( sanitize_title( $p[0] ), OBJECT, 'post' ) ) continue;
        $id = wp_insert_post( array( 'post_type' => 'post', 'post_title' => $p[0], 'post_content' => $p[2], 'post_excerpt' => $p[1], 'post_status' => 'publish' ) );
        if ( ! is_wp_error( $id ) ) {
            wp_set_object_terms( $id, $p[3], 'category', true );
            wp_set_object_terms( $id, $p[4], 'post_tag' );
        }
    }

    // --- PERMALINK STRUCTURE ---
    global $wp_rewrite;
    $wp_rewrite->set_permalink_structure( '/%postname%/' );
    $wp_rewrite->flush_rules();

    update_option( 'fresh_demo_installed', true );
}
add_action( 'after_switch_theme', 'fresh_activation' );
